<?php
/*
 * Plugin Name: WordPress Maintenance
 * Version: 1.4.6
 * Plugin URI: https://webd.uk/support/
 * Description: Maintenance tools for WordPress
 * Author: Webd Ltd
 * Author URI: https://webd.uk
 * Text Domain: wordpress-maintenance
 * Update URI: https://webd.uk/product/wordpress-maintenance-upgrade/
 */



if (!defined('ABSPATH')) {
    exit('This isn\'t the page you\'re looking for. Move along, move along.');
}



if (!class_exists('wordpress_maintenance_class')) {

	class wordpress_maintenance_class {

        public static $version = '1.4.6';
        public $load_average = false;
        public $cores_available = false;
        public $cores_recommended = false;
        public $server_overloaded = false;
        private $content_changed = false;

		function __construct() {

            // Analyse CPUs and Load Average
            $this->analyse_load_average();

            // Clear Memcached cache after an update
            add_action('upgrader_post_install', 'wordpress_maintenance_class::upgrader_post_install', 10, 3);

            // Detect active options
            add_action('add_option', 'wordpress_maintenance_class::add_option', 10, 2);
            add_filter('pre_option', 'wordpress_maintenance_class::pre_option', 10, 3);
            add_filter('pre_update_option', 'wordpress_maintenance_class::pre_option', 10, 3);
            add_action('shutdown', array($this, 'shutdown'));

            // Enable Database Tools and prevent Flamingo saving contacts
            // Test for StackCache and enable auto-purge on content change
            add_action('plugins_loaded', array($this, 'plugins_loaded'));

            if (is_admin()) {

                add_action('admin_menu', array($this, 'wordpress_maintenance_menu'));
                add_action('wp_ajax_wordpress_maintenance_toggle_debug_log', array($this,'wordpress_maintenance_toggle_debug_log'));
                add_action('wp_ajax_wordpress_maintenance_delete_empty_wp_cron_events', array($this,'wordpress_maintenance_delete_empty_wp_cron_events'));
                add_action('wp_ajax_wordpress_maintenance_delete_inactive_widgets', array($this,'wordpress_maintenance_delete_inactive_widgets'));
                add_action('wp_ajax_wordpress_maintenance_delete_languages', array($this,'wordpress_maintenance_delete_languages'));
                add_action('wp_ajax_wordpress_maintenance_delete_posts', array($this,'wordpress_maintenance_delete_posts'));
                add_action('wp_ajax_wordpress_maintenance_delete_comments', array($this,'wordpress_maintenance_delete_comments'));
                add_action('wp_ajax_wordpress_maintenance_close_comments', array($this,'wordpress_maintenance_close_comments'));
                add_action('wp_ajax_wordpress_maintenance_close_pings', array($this,'wordpress_maintenance_close_pings'));
                add_action('wp_ajax_wordpress_maintenance_toggle_wp_cron', array($this,'wordpress_maintenance_toggle_wp_cron'));
                add_action('wp_ajax_wordpress_maintenance_toggle_file_edit', array($this,'wordpress_maintenance_toggle_file_edit'));
                add_action('wp_ajax_wordpress_maintenance_enable_updates', array($this,'wordpress_maintenance_enable_updates'));
                add_action('wp_ajax_wordpress_maintenance_disable_updates', array($this,'wordpress_maintenance_disable_updates'));
                add_action('wp_ajax_wordpress_maintenance_enable_service_notifications', array($this,'wordpress_maintenance_enable_service_notifications'));
                add_action('wp_ajax_wordpress_maintenance_disable_service_notifications', array($this,'wordpress_maintenance_disable_service_notifications'));
                add_action('wp_ajax_wordpress_maintenance_disable_links', array($this,'wordpress_maintenance_disable_links'));
                add_action('wp_ajax_wordpress_maintenance_delete_upload_path_option', array($this,'wordpress_maintenance_delete_upload_path_option'));
                add_action('wp_ajax_wordpress_maintenance_convert_tables_to_innodb', array($this,'wordpress_maintenance_convert_tables_to_innodb'));
                add_action('wp_ajax_wordpress_maintenance_change_admin_email', array($this,'wordpress_maintenance_change_admin_email'));
                add_action('wp_ajax_wordpress_maintenance_set_revision_retention', array($this,'wordpress_maintenance_set_revision_retention'));
                add_action('wp_ajax_wordpress_maintenance_set_cloudflare_api_token', array($this,'wordpress_maintenance_set_cloudflare_api_token'));
                add_action('wp_ajax_wordpress_maintenance_recover_custom_css', array($this,'wordpress_maintenance_recover_custom_css'));
                add_action('wp_ajax_wordpress_maintenance_deautoload_option', array($this,'wordpress_maintenance_deautoload_option'));
                add_action('wp_ajax_wordpress_maintenance_delete_option', array($this,'wordpress_maintenance_delete_option'));
                add_action('wp_ajax_wordpress_maintenance_delete_option_group', array($this,'wordpress_maintenance_delete_option_group'));
                add_action('wp_ajax_wordpress_maintenance_delete_all_transients', array($this,'wordpress_maintenance_delete_all_transients'));
                add_action('wp_ajax_wordpress_maintenance_delete_oembed_cache_postmeta', array($this,'wordpress_maintenance_delete_oembed_cache_postmeta'));
                add_action('wp_ajax_wordpress_maintenance_download_orphaned_meta_data', array($this,'wordpress_maintenance_download_orphaned_meta_data'));
                add_action('wp_ajax_wordpress_maintenance_delete_orphaned_meta_data', array($this,'wordpress_maintenance_delete_orphaned_meta_data'));
                add_action('wp_ajax_wordpress_maintenance_set_cloudflare_security_level', array($this,'wordpress_maintenance_set_cloudflare_security_level'));
                add_action('wp_ajax_wordpress_maintenance_flush_persistent_object_cache', array($this,'wordpress_maintenance_flush_persistent_object_cache'));

				add_action('wp_ajax_dismiss_wordpress_maintenance_notice_handler', 'wmCommon::ajax_notice_handler');
                add_filter('update_plugins_webd.uk', 'wmPremium::check_plugin_update', 10, 4);
                add_filter('plugins_api', 'wmPremium::get_plugin_information', 10, 3);
                add_filter('after_plugin_row_' . plugin_basename(__FILE__), 'wmPremium::activate_purchase_js', 10, 3);
                add_action('wp_ajax_wordpress_maintenance_activate_purchase', 'wmPremium::activate_purchase');
                add_action('admin_notices', 'wmPremium::upgrade_notice');
                add_filter('upgrader_pre_download', 'wmPremium::upgrader_pre_download', 10, 3);
                add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'wordpress_maintenance_add_plugin_settings_links'));

                add_action('wp_dashboard_setup', array($this, 'wp_dashboard_setup'));
                add_action('admin_notices', array($this, 'admin_notices'));

                add_action('show_user_profile', array($this, 'edit_user_profile'));
                add_action('edit_user_profile', array($this, 'edit_user_profile'));
                add_action('personal_options_update', array($this, 'edit_user_profile_update'));
                add_action('edit_user_profile_update', array($this, 'edit_user_profile_update'));

            }

            add_action('cron_schedules', 'wordpress_maintenance_class::cron_schedules');
            add_action('wordpress_maintenance_daily', array($this, 'wordpress_maintenance_daily_schedule'));

            if (!wp_next_scheduled('wordpress_maintenance_daily')) {

                wp_schedule_event(time(), 'daily', 'wordpress_maintenance_daily');

            }

            add_action('wordpress_maintenance_hourly', array($this, 'wordpress_maintenance_hourly_schedule'));

            if (!wp_next_scheduled('wordpress_maintenance_hourly')) {

                wp_schedule_event(time(), 'hourly', 'wordpress_maintenance_hourly');

            }

            add_action('wordpress_maintenance_quarter_hourly', array($this, 'wordpress_maintenance_quarter_hourly_schedule'));

            if (!wp_next_scheduled('wordpress_maintenance_quarter_hourly')) {

                wp_schedule_event(time(), 'fifteen_minutes', 'wordpress_maintenance_quarter_hourly');

            }

            add_action('wordfence_security_event', 'wordpress_maintenance_class::wordfence_security_event', 9);

		}

        public function analyse_load_average() {

            $this->load_average = sys_getloadavg();

            if ($this->load_average && isset($this->load_average[2])) {

				if (is_file('/proc/cpuinfo')) {

					preg_match_all('/^processor/m', file_get_contents('/proc/cpuinfo'), $matches);
					$this->cores_available = count($matches[0]);

				} elseif ('WIN' === strtoupper(substr(PHP_OS, 0, 3))) {

					$process = @popen('wmic cpu get NumberOfCores', 'rb');

					if (false !== $process) {

						fgets($process);
						$this->cores_available = intval(fgets($process));
						pclose($process);

					}

				} else {

					$process = @popen('sysctl -a', 'rb');

					if (false !== $process) {

						$output = stream_get_contents($process);

						preg_match('/hw.ncpu: (\d+)/', $output, $matches);

						if ($matches) {

							$this->cores_available = intval($matches[1][0]);

						}

					}

					pclose($process);

                    if (
					    !$this->cores_available &&
					    function_exists('shell_exec')
				    ) {

						$process = (int)(rtrim(@shell_exec('nproc')));

						if ($process) {

							$this->cores_available = $process;

						}

					}

				}

                if ($this->cores_available) {

                    if ($this->load_average[2]) {

                        $this->cores_recommended = ceil(($this->load_average[2] / 3) * 4);

                    } else {

                        $this->cores_recommended = 1;

                    }

                    if ($this->load_average[0] / $this->cores_available > 1) {

                        $this->server_overloaded = true;

                    }

                    if (sanitize_text_field(get_option('wm_cloudflare_api_token'))) {

                        add_action('init', array($this, 'check_load_average'));

                    }

                } else {

                    $this->cores_available = false;

                }

            } else {

                $this->load_average = false;

            }

        }

        public function check_load_average() {

            $under_attack_status = self::check_under_attack_file();

            if (
                !$under_attack_status &&
                $this->load_average &&
                $this->load_average[0] &&
                $this->load_average[1] &&
                $this->cores_available &&
                (
                    (
                        ($this->load_average[0] / $this->cores_available > 1) &&
                        ($this->load_average[1] / $this->cores_available > 1)
                    ) ||
                    (
                        ($this->load_average[0] / $this->cores_available > 2)
                    )
                ) &&
                self::acquire_under_attack_lock()
            ) {

                $required_cpus = ceil(max($this->load_average[0], $this->load_average[1], $this->load_average[2]));

                $message = __('The website\'s server is overloaded so Cloudflare "Under Attack" mode has been enabled.', 'wordpress-maintenance') . '

' . sprintf(__('The server’s traffic requires %d CPUs but you only have %d CPUs - if this happens frequently you may need to upgrade your server.', 'wordpress-maintenance'), $required_cpus, $this->cores_available);

                if (!self::check_under_attack_file()) {

                    self::enable_cloudflare_under_attack_mode($message, true);

                }

                self::release_under_attack_lock(true);

            } elseif ($under_attack_status) {

                if (
                    (
                        $under_attack_status['overloaded'] ||
                        (
                            !$under_attack_status['overloaded'] &&
                            $under_attack_status['timestamp'] < (time() - 900)
                        )
                    ) &&
                    $this->load_average &&
                    $this->load_average[0] &&
                    $this->load_average[1] &&
                    $this->load_average[2] &&
                    $this->cores_available &&
                    ($this->load_average[0] / $this->cores_available < 0.75) &&
                    ($this->load_average[1] / $this->cores_available < 0.75) &&
                    ($this->load_average[2] / $this->cores_available < 0.75) &&
                    self::acquire_under_attack_lock()
                ) {

                    if ($under_attack_status) {

                        self::disable_cloudflare_under_attack_mode();

                    }

                    self::release_under_attack_lock(false);

                }

            }

        }

        private static function create_under_attack_file($overloaded = false) {

            if (!function_exists('WP_Filesystem')) {

                require_once ABSPATH . 'wp-admin/includes/file.php';

            }

            WP_Filesystem();

    		global $wp_filesystem;

        	$under_attack_file = ABSPATH . '.under_attack';

            $under_attack_data = array(
                'timestamp' => time(),
                'overloaded' => $overloaded
            );

        	$wp_filesystem->delete($under_attack_file);
        	$wp_filesystem->put_contents($under_attack_file, json_encode($under_attack_data), FS_CHMOD_FILE);

            return file_exists($under_attack_file);

        }

        private static function delete_under_attack_file() {

            if (!function_exists('WP_Filesystem')) {

                require_once ABSPATH . 'wp-admin/includes/file.php';

            }

            WP_Filesystem();

    		global $wp_filesystem;

        	$under_attack_file = ABSPATH . '.under_attack';
        	$wp_filesystem->delete($under_attack_file);

            return !file_exists($under_attack_file);

        }

        private static function check_under_attack_file() {

        	$under_attack_file = ABSPATH . '.under_attack';
            $under_attack_data = false;

            if (file_exists($under_attack_file)) {

                $under_attack_data = file_get_contents($under_attack_file);

                if ($under_attack_data) {

                    $under_attack_data = json_decode($under_attack_data, true);

                    if (
                        is_array($under_attack_data) &&
                        isset($under_attack_data['timestamp']) &&
                        isset($under_attack_data['overloaded'])
                    ) {

                        $under_attack_data = array(
                            'timestamp' => absint($under_attack_data['timestamp']),
                            'overloaded' => (bool) $under_attack_data['overloaded']
                        );

                    } else {

                        $under_attack_data = false;

                    }

                } else {

                    $under_attack_data = false;

                }

                if (!$under_attack_data) {

                    if (!self::delete_under_attack_file()) {

                        error_log('Could not delete .under_attack');

                    }

                }

            }

            return $under_attack_data;

        }

        private static function acquire_under_attack_lock() {

            global $wpdb;

            return (bool) $wpdb->get_var($wpdb->prepare("SELECT GET_LOCK(%s, %d)", 'wm_under_attack_lock', 0));

        }

        private static function release_under_attack_lock($required_status) {

            $max_attempts = 5;
            $attempt = 0;
            $status_confirmed = false;

            while (
                !$status_confirmed &&
                $attempt < $max_attempts
            ) {

                $current_status = (bool) self::check_under_attack_file();

                if (
                    $current_status === $required_status
                ) {

                    $status_confirmed = true;

                } else {

                    usleep(200000);

                }

                $attempt++;

            }

            global $wpdb;

            $wpdb->get_var($wpdb->prepare("SELECT RELEASE_LOCK(%s)", 'wm_under_attack_lock'));

            error_log('Under attack ' . ($required_status ? 'enable' : 'disable') . ' status ' . ($status_confirmed ? '' : 'not ') . 'confirmed in ' . $attempt . ' attempts.');

        }

		function wordpress_maintenance_add_plugin_settings_links($links) {

			$settings_links = wmCommon::plugin_action_links(add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')), true);

			return array_merge($settings_links, $links);

		}

        public function plugins_loaded() {

            global $current_user;

            if ($this->request_permission() && current_user_can('manage_options')) {

                define('WP_ALLOW_REPAIR', true);

            }

            if (defined('FLAMINGO_VERSION')) {

                add_filter('flamingo_add_contact', '__return_empty_string');

            }

            global $wpsc;

            if (
                isset($wpsc) &&
                method_exists($wpsc, 'purge')
            ) {

                add_action('save_post', array($this, 'wm_content_changed'), 10, 2);
                add_action('edited_term', array($this, 'wm_content_changed'), 10, 2);

            }

        }

        public function wm_content_changed($post_id, $post) {

            if (
                'save_post' === current_action() &&
                $post instanceof WP_Post &&
                isset($post->post_status) &&
                in_array($post->post_status, array(
                    'inherit',
                    'publish',
                    'private'
                )) &&
                get_post_types(array(
                    'name' => $post->post_type,
                    'exclude_from_search' => false
                ))
            ) {

                $this->content_changed = true;

            } elseif ('edited_term' === current_action()) {

                $this->content_changed = true;

            }

        }

        public function wordpress_maintenance_menu() {

            add_submenu_page('tools.php', __('WordPress Maintenance', 'wordpress-maintenance'), __('Maintenance', 'wordpress-maintenance'), 'manage_options', 'wordpress-maintenance', array($this, 'wordpress_maintenance_page'));

        }

        public function wordpress_maintenance_page() {

?>
<div class="wrap">
<h1><?php _e('WordPress Maintenance', 'wordpress-maintenance'); ?></h1>
<?php

            $current_tab = 'maintenance';

            $tabs = array(
                'maintenance' => 'Maintenance',
                'settings' => 'Settings',
                'phpinfo' => 'PHP Info'
            );

?>
<h2 class="nav-tab-wrapper">
<?php

            foreach ($tabs as $tab => $title) {

                $class = ($current_tab === $tab) ? ' nav-tab-active' : '';

?>
<a id="<?php echo esc_attr($tab); ?>" class="nav-tab<?php echo $class; ?>" href="#" title="<?php echo esc_attr($title); ?>"><?php echo esc_html($title); ?></a>
<?php

            }

?>
</h2>
<script type="text/javascript">
(function($) {
    $('.nav-tab-wrapper .nav-tab').click(function() {
        $('.tab_content').hide();
        $('.tab_content.' + $(this).attr('id')).show();
        $('.nav-tab-active').removeClass('nav-tab-active');
        $(this).addClass('nav-tab-active');
    });
})(jQuery);
</script>
<?php

            global $wpdb;

            if (!wmPremium::request_permission()) {

?>
<div class="notice notice-info">
<p><a href="<?php echo wmCommon::upgrade_link(); ?>" title="<?php _e('Purchase ' . wmCommon::plugin_name(), wmCommon::plugin_text_domain()); ?>">Purchase a license to use these features.</a> WordPress Maintenance only costs &pound;9.99 and every purchase helps fund further development!</p>
</div>
<?php

            }

?>
<p>Make sure you <strong>take a full filesystem and database backup of your site</strong> before you use any of these tools!</p>
<div class="wrap tab_content maintenance"<?php echo ('maintenance' === $current_tab ? '' : ' style="display: none;"'); ?>>
<h2><?php _e('Maintenance', 'template-editor'); ?></h2>
<?php

            if (-1 === get_theme_mod('custom_css_post_id')) {

                $css_found = false;

                foreach (get_posts(array(
    'post_type' => 'custom_css',
    'numberposts' => -1
)) as $custom_css) {

                    if (get_stylesheet() === $custom_css->post_title) {

                        $css_found = true;

                    }

                }

                if ($css_found) {

?>
<p><span class="wordpress-maintenance-recover-custom-css button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Recover Custom CSS', 'wordpress-maintenance'); ?></span> <?php _e('Recover custom CSS that has been located for the active theme', 'wordpress-maintenance'); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-recover-custom-css').click(function() {
        if (confirm('<?php _e('Are you sure you want to recover custom CSS?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-recover-custom-css').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_recover_custom_css',
            	_ajax_nonce: '<?php echo wp_create_nonce('recover-custom-css'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

                }

            }

		    $language_issues = array(
                'locales' => array(),
	            'plugins' => array(),
	            'themes' => array()
            );

            if (is_dir(WP_CONTENT_DIR . '/languages/')) {

    		    $iterator = new RecursiveDirectoryIterator(WP_CONTENT_DIR . '/languages/');

        		foreach (new RecursiveIteratorIterator($iterator, RecursiveIteratorIterator::CHILD_FIRST) as $file) {

        			$file_info = pathinfo($file->getPathname());

        			if ($file->isFile() && isset($file_info['extension']) && in_array(strtolower($file_info['extension']), array('mo', 'po', 'json'))) {

                        $language = $file_info['filename'];

                        if (false !== strpos($language, '-')) {

                            if (in_array($file_info['extension'], array('mo', 'po'))) {

                                $directory = rtrim($file_info['dirname'],'/');
                                $directory = substr($directory, strrpos($directory, '/') + 1);

                                if ('languages' !== $directory) {

                                    $theme_or_plugin = substr($language, 0, strrpos($language, '-'));

                                    if (!is_dir(WP_CONTENT_DIR . '/' . $directory . '/' . $theme_or_plugin)) {

                                        if (!isset($language_issues[$directory][$theme_or_plugin])) {

                                            $language_issues[$directory][$theme_or_plugin] = 1;

                                        } else {

                                            $language_issues[$directory][$theme_or_plugin] = $language_issues[$directory][$theme_or_plugin] + 1;

                                        }

                                        if (!isset($language_issues[$directory]['total'])) {

                            				$language_issues[$directory]['total'] = 1;

                                        } else {

                            				$language_issues[$directory]['total'] = $language_issues[$directory]['total'] + 1;

                                        }

                                    }

                                }

                                $language = substr($language, strrpos($language, '-') + 1);

                            } else {

                                $language = preg_replace('/-wc-admin-app$/', '-woocommerce', $language);
                                $language = substr($language, 0, strrpos($language, '-'));
    
                                if (false !== strpos($language, '-')) {

                                    $language = substr($language, strrpos($language, '-') + 1);

                                }

                            }

                        }

                        if ($language !== get_locale()) {

                            if (!isset($language_issues['locales'][$language])) {

                				$language_issues['locales'][$language] = 1;

                            } else {

                				$language_issues['locales'][$language] = $language_issues['locales'][$language] + 1;

                            }

                            if (!isset($language_issues['locales']['total'])) {

                				$language_issues['locales']['total'] = 1;

                            } else {

                				$language_issues['locales']['total'] = $language_issues['locales']['total'] + 1;

                            }

                        }

        			}

        		}

            }

            if ($language_issues['locales']) {

?>
<p><span class="wordpress-maintenance-delete-language-locales button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Unused Languages', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d unused language locale file', 'Delete %d unused language locale files', $language_issues['locales']['total'], 'wordpress-maintenance'), $language_issues['locales']['total']) ?> - <?php

    foreach ($language_issues['locales'] as $key => $value) {

        if ('total' !== $key) {

            echo ' Language: ' . esc_html($key) . ' (' . $value . ')';

        }

    }

?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-language-locales').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete unused languages?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-language-locales').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_languages',
            	file_type: 'locales',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-languages-locales'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            if ($language_issues['plugins']) {

?>
<p><span class="wordpress-maintenance-delete-language-plugins button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Plugin Languages', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d deleted plugin language file', 'Delete %d deleted plugin language files', $language_issues['plugins']['total'], 'wordpress-maintenance'), $language_issues['plugins']['total']) ?> - <?php

    foreach ($language_issues['plugins'] as $key => $value) {

        if ('total' !== $key) {

            echo ' Plugin: ' . esc_html($key) . ' (' . $value . ')';

        }

    }

?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-language-plugins').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete plugin languages?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-language-plugins').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_languages',
            	file_type: 'plugins',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-languages-plugins'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            if ($language_issues['themes']) {

?>
<p><span class="wordpress-maintenance-delete-language-themes button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Theme Languages', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d deleted theme language file', 'Delete %d deleted theme language files', $language_issues['themes']['total'], 'wordpress-maintenance'), $language_issues['themes']['total']) ?> - <?php

    foreach ($language_issues['themes'] as $key => $value) {

        if ('total' !== $key) {

            echo ' Theme: ' . esc_html($key) . ' (' . $value . ')';

        }

    }

?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-language-themes').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete theme languages?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-language-themes').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_languages',
            	file_type: 'themes',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-languages-themes'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $empty_event_hooks = $this->wordpress_maintenance_get_empty_cron_event_hooks();

            if ($empty_event_hooks) {

?>
<p><span class="wordpress-maintenance-delete-empty-wp-cron-events button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Empty WP CRON Events', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d scheduled event hook (%s) in WP CRON that have no associated actions.', 'Delete %d scheduled event hooks (%s) in WP CRON that have no associated actions.', count($empty_event_hooks), 'wordpress-maintenance'), count($empty_event_hooks), implode(', ', $empty_event_hooks)); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-empty-wp-cron-events').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete empty WP CRON events?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-empty-wp-cron-events').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_empty_wp_cron_events',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-empty-wp-cron-events'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $sidebars_widgets = wp_get_sidebars_widgets();
            $inactive_widget_count = 0;
            $legacy_widget_count = 0;

            if (is_array($sidebars_widgets) && $sidebars_widgets) {

                foreach ($sidebars_widgets as $sidebar_id => $widgets) {

                    foreach ($widgets as $widget) {

                        if (0 !== substr_compare($widget, 'block-', 0, 6)) {

                            $legacy_widget_count++;

                        }

                    }

                	if (!('wp_inactive_widgets' !== $sidebar_id && is_registered_sidebar($sidebar_id)) && $widgets) {

                    	$inactive_widget_count += count($widgets);

                	}

                }

            }

            if ($inactive_widget_count) {

?>
<p><span class="wordpress-maintenance-delete-inactive-widgets button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Inactive Widgets', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d inactive widget that does not have an associated sidebar.', 'Delete %d inactive widgets that do not have an associated sidebar.', $inactive_widget_count, 'wordpress-maintenance'), $inactive_widget_count); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-inactive-widgets').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete inactive widgets?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-inactive-widgets').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_inactive_widgets',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-inactive-widgets'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            if ($legacy_widget_count) {

?>
<p><a class="button button-small" title="<?php _e('Edit Widgets', 'wordpress-maintenance'); ?>" href="<?php echo esc_url(admin_url('widgets.php')); ?>"><?php _e('Edit Widgets', 'wordpress-maintenance'); ?></a> <?php printf(_n('You have a legacy widget. ', 'You have %d legacy widgets. ', $legacy_widget_count, 'wordpress-maintenance'), $legacy_widget_count); ?><?php _e('Legacy widgets cause issues in the Block Widget editor due to a <a href="https://core.trac.wordpress.org/ticket/53801">bug whereby the wp_footer hook is fired</a>. You should convert them to Block Widgets.', 'wordpress-maintenance'); ?></p>
<?php

            }

            $flamingo_contacts = get_posts(array(
                'post_type' => 'flamingo_contact',
                'numberposts' => -1,
                'fields' => 'ids'
            ));

            if ($flamingo_contacts) {

?>
<p><span class="wordpress-maintenance-delete-posts-contacts button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Flamingo Contacts', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d Flamingo contact.', 'Delete %d Flamingo contacts.', count($flamingo_contacts), 'wordpress-maintenance'), count($flamingo_contacts)) ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-posts-contacts').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete Flamingo contacts?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-posts-contacts').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_posts',
            	post_type: 'flamingo_contact',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-posts-flamingo_contact'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $flamingo_messages = get_posts(array(
                'post_type' => 'flamingo_inbound',
                'numberposts' => -1,
                'fields' => 'ids'
            ));

            if ($flamingo_messages) {

?>
<p><span class="wordpress-maintenance-delete-posts-messages button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Flamingo Messages', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d Flamingo message.', 'Delete %d Flamingo messages.', count($flamingo_messages), 'wordpress-maintenance'), count($flamingo_messages)) ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-posts-messages').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete Flamingo messages?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-posts-messages').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_posts',
            	post_type: 'flamingo_inbound',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-posts-flamingo_inbound'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $flamingo_messages = get_posts(array(
                'post_type' => 'flamingo_inbound',
                'post_status' => 'flamingo-spam',
                'numberposts' => -1,
                'fields' => 'ids'
            ));

            if ($flamingo_messages) {

?>
<p><span class="wordpress-maintenance-delete-posts-spam button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Flamingo Spam', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d Flamingo spam message.', 'Delete %d Flamingo spam messages.', count($flamingo_messages), 'wordpress-maintenance'), count($flamingo_messages)) ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-posts-spam').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete Flamingo spam messages?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-posts-spam').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_posts',
            	post_type: 'flamingo_inbound',
            	post_status: 'flamingo-spam',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-posts-flamingo_inbound'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $flamingo_messages = get_posts(array(
                'post_type' => 'flamingo_inbound',
                'post_status' => 'trash',
                'numberposts' => -1,
                'fields' => 'ids'
            ));

            if ($flamingo_messages) {

?>
<p><span class="wordpress-maintenance-delete-posts-trash button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Flamingo Trash', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d Flamingo trash message.', 'Delete %d Flamingo trash messages.', count($flamingo_messages), 'wordpress-maintenance'), count($flamingo_messages)) ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-posts-trash').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete Flamingo trash messages?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-posts-trash').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_posts',
            	post_type: 'flamingo_inbound',
            	post_status: 'trash',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-posts-flamingo_inbound'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $revisions = get_posts(array(
                'post_type' => 'revision',
                'post_status' => 'inherit',
                'numberposts' => -1,
                'fields' => 'ids'
            ));

            if ($revisions && !absint(get_option('wm_revision_retention'))) {

?>
<p><span class="wordpress-maintenance-delete-revisions button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Revisions', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d post revision.', 'Delete %d post revisions.', count($revisions), 'wordpress-maintenance'), count($revisions)) ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-revisions').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete post revisions?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-revisions').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_posts',
            	post_type: 'revision',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-posts-revision'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $auto_drafts = get_posts(array(
                'post_type' => 'any',
                'post_status' => 'auto-draft',
                'numberposts' => -1,
                'fields' => 'ids'
            ));

            if (
                $auto_drafts &&
                count($auto_drafts) > 2
            ) {

?>
<p><span class="wordpress-maintenance-delete-auto-drafts button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Auto Drafts', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d auto draft.', 'Delete %d auto drafts.', count($auto_drafts), 'wordpress-maintenance'), count($auto_drafts)) ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-auto-drafts').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete auto drafts?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-auto-drafts').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_posts',
            	post_type: 'any',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-posts-any'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $oembed_caches = get_posts(array(
                'post_type' => 'any',
                'numberposts' => -1,
                'meta_query' => array(array(
                   'key'     => '_oembed_',
                   'compare_key' => 'LIKE'
                )),
                'fields' => 'ids'
            ));

            if ($oembed_caches) {

?>
<p><span class="wordpress-maintenance-delete-oembed-cache-postmeta button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete oEmbed Cache Meta', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete oEmbed Cache post meta from %d post.', 'Delete oEmbed Cache post meta from %d posts.', count($oembed_caches), 'wordpress-maintenance'), count($oembed_caches)) ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-oembed-cache-postmeta').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete oEmbed Cache postmeta?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-oembed-cache-postmeta').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_oembed_cache_postmeta',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-oembed-cache-postmeta'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $oembed_caches = get_posts(array(
                'post_type' => 'oembed_cache',
                'numberposts' => -1,
                'fields' => 'ids'
            ));

            if ($oembed_caches) {

?>
<p><span class="wordpress-maintenance-delete-oembed-caches button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete oEmbed Caches', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d oEmbed Cache.', 'Delete %d oEmbed Caches.', count($oembed_caches), 'wordpress-maintenance'), count($oembed_caches)) ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-oembed-caches').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete oEmbed Caches?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-oembed-caches').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_posts',
            	post_type: 'oembed_cache',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-posts-oembed_cache'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $comments = get_comments(array(
                'status' => 'hold',
                'count' => true
            ));

            if ($comments) {

?>
<p><span class="wordpress-maintenance-delete-hold-comments button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Pending Comments', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d pending comment.', 'Delete %d pending comments.', $comments, 'wordpress-maintenance'), $comments); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-hold-comments').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete pending comments?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-hold-comments').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_comments',
            	status: 'hold',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-hold-comments'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $comments = get_comments(array(
                'status' => 'spam',
                'count' => true
            ));

            if ($comments) {

?>
<p><span class="wordpress-maintenance-delete-spam-comments button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Spam Comments', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d spam comment.', 'Delete %d spam comments.', $comments, 'wordpress-maintenance'), $comments); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-spam-comments').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete spam comments?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-spam-comments').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_comments',
            	status: 'spam',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-spam-comments'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $comments = get_comments(array(
                'status' => 'trash',
                'count' => true
            ));

            if ($comments) {

?>
<p><span class="wordpress-maintenance-delete-trash-comments button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Empty Comments Bin', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d comment from the bin.', 'Delete %d comments from the bin.', $comments, 'wordpress-maintenance'), $comments); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-trash-comments').click(function() {
        if (confirm('<?php _e('Are you sure you want to empty the comments bin?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-trash-comments').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_comments',
            	status: 'trash',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-trash-comments'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            if ('closed' === get_option('default_comment_status')) {

                $comments = get_comments(array(
                    'status' => 'all',
                    'count' => true,
                    'type__not_in' => array('order_note', 'review')
                ));

                if ($comments) {

?>
<p><span class="wordpress-maintenance-delete-all-comments button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>" style="color: red;"><?php _e('Delete All Comments', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d comment.', 'Delete %d comments.', $comments, 'wordpress-maintenance'), $comments); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-all-comments').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete all comments?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-all-comments').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_comments',
            	status: 'all',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-all-comments'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

                }

            }

            $posts = get_posts(array(
                'comment_status' => 'open',
                'posts_per_page' => -1,
                'post_type' => 'any'
            ));

            if ($posts || 'open' === get_option('default_comment_status')) {

?>
<p><span class="wordpress-maintenance-close-comments button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Close Comments', 'wordpress-maintenance'); ?></span> <?php printf(_n('Close comments on %d post and future posts.', 'Close comments on %d posts and future posts.', count($posts), 'wordpress-maintenance'), count($posts)); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-close-comments').click(function() {
        if (confirm('<?php _e('Are you sure you want to close comments on all posts?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-close-comments').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_close_comments',
            	_ajax_nonce: '<?php echo wp_create_nonce('close-comments'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $posts = get_posts(array(
                'ping_status' => 'open',
                'posts_per_page' => -1,
                'post_type' => 'any'
            ));

            if ($posts || 'open' === get_option('default_ping_status')) {

?>
<p><span class="wordpress-maintenance-close-pings button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Close Pings', 'wordpress-maintenance'); ?></span> <?php printf(_n('Close pingbacks and trackbacks on %d post and future posts.', 'Close pingbacks and trackbacks on %d posts and future posts.', count($posts), 'wordpress-maintenance'), count($posts)); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-close-pings').click(function() {
        if (confirm('<?php _e('Are you sure you want to close pingbacks and trackbacks on all posts?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-close-pings').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_close_pings',
            	_ajax_nonce: '<?php echo wp_create_nonce('close-pings'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            if (get_option('link_manager_enabled', true)) {

?>
<p><span class="wordpress-maintenance-disable-links button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Disable Links', 'wordpress-maintenance'); ?></span> <?php _e('Delete existing links and disable link manager.', 'wordpress-maintenance'); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-disable-links').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete links and dissable link manager?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-disable-links').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_disable_links',
            	_ajax_nonce: '<?php echo wp_create_nonce('disable-links'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            if (get_option('upload_path')) {

?>
<p><span class="wordpress-maintenance-delete-upload-path-option button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Upload Path Option', 'wordpress-maintenance'); ?></span> <?php _e('Delete the upload_path option from the wp_options table in the database.', 'wordpress-maintenance'); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-upload-path-option').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete the upload_path option?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-upload-path-option').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_upload_path_option',
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-upload-path-option'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $database_tables = $wpdb->get_results("SHOW TABLE STATUS");
            $myisam_tables = array();

            foreach ($database_tables as $key => $database_table) {

                if (
                    'InnoDB' !== $database_table->Engine && 
                    'MEMORY' !== $database_table->Engine
                ) {

                    $myisam_tables[] = $database_table->Name;

                }

            }

            if ($myisam_tables) {

?>
<p><span class="wordpress-maintenance-convert-tables-to-innodb button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Convert Tables to InnoDB', 'wordpress-maintenance'); ?></span> <?php printf(__('Convert the MyISAM tables (%s) in the database to InnoDB.', 'wordpress-maintenance'), implode(',', $myisam_tables)); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-convert-tables-to-innodb').click(function() {
        if (confirm('<?php _e('Are you sure you want to convert the database tables to InnoDB?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-convert-tables-to-innodb').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_convert_tables_to_innodb',
            	_ajax_nonce: '<?php echo wp_create_nonce('convert-tables-to-innodb'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

?>
<p><a class="button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"<?php if ($this->request_permission()) { echo ' target="_blank"'; } ?> href="<?php echo $this->request_permission() ? admin_url('maint/repair.php') : '#'; ?>"><?php echo esc_html__('Open Database Tools', 'wordpress-maintenance'); ?></a> <?php _e('Open WordPress database repair and optimize tools.', 'wordpress-maintenance'); ?></p>
<p>Admin Email: <input type="email" id="wordpress-maintenance-admin-email" value="<?php echo esc_attr(get_option('admin_email')); ?>" /><br />
<span class="wordpress-maintenance-change-admin-email button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Change Admin Email', 'wordpress-maintenance'); ?></span> <?php _e('Change admin email in general settings without requiring a confirmation email.', 'wordpress-maintenance'); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-change-admin-email').click(function() {
        if (confirm('<?php _e('Are you sure you want to change the admin email?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-change-admin-email').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_change_admin_email',
            	_ajax_nonce: '<?php echo wp_create_nonce('change-admin-email'); ?>',
            	email: $('#wordpress-maintenance-admin-email').val()
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            $orphaned_meta_checks = array(
                array(
                    'table' => $wpdb->comments,
                    'id' => 'comment_ID',
                    'meta_table' => $wpdb->commentmeta,
                    'meta_id' => 'meta_id',
                    'meta_object_id' => 'comment_id'
                ),
                array(
                    'table' => $wpdb->posts,
                    'id' => 'ID',
                    'meta_table' => $wpdb->postmeta,
                    'meta_id' => 'meta_id',
                    'meta_object_id' => 'post_id'
                ),
                array(
                    'table' => $wpdb->terms,
                    'id' => 'term_id',
                    'meta_table' => $wpdb->termmeta,
                    'meta_id' => 'meta_id',
                    'meta_object_id' => 'term_id'
                ),
                array(
                    'table' => $wpdb->users,
                    'id' => 'ID',
                    'meta_table' => $wpdb->usermeta,
                    'meta_id' => 'umeta_id',
                    'meta_object_id' => 'user_id'
                ),
            );

            $orphaned_meta_check_results = array();

            foreach ($orphaned_meta_checks as $orphaned_meta_check) {

                $orphaned_meta = $wpdb->get_results("SELECT COUNT(*) AS meta_count FROM {$orphaned_meta_check['meta_table']} LEFT JOIN {$orphaned_meta_check['table']} ON {$orphaned_meta_check['meta_table']}.{$orphaned_meta_check['meta_object_id']} = {$orphaned_meta_check['table']}.{$orphaned_meta_check['id']} WHERE {$orphaned_meta_check['table']}.{$orphaned_meta_check['id']} IS NULL;");

                if (isset($orphaned_meta[0]->meta_count) && absint($orphaned_meta[0]->meta_count)) {

                    $orphaned_meta_check_results[$orphaned_meta_check['meta_table']] = array(
                        'total' => $orphaned_meta[0]->meta_count,
                        'meta_keys' => $wpdb->get_results("SELECT meta_key FROM {$orphaned_meta_check['meta_table']} LEFT JOIN {$orphaned_meta_check['table']} ON {$orphaned_meta_check['meta_table']}.{$orphaned_meta_check['meta_object_id']} = {$orphaned_meta_check['table']}.{$orphaned_meta_check['id']} WHERE {$orphaned_meta_check['table']}.{$orphaned_meta_check['id']} IS NULL GROUP BY meta_key;")
                    );

                    foreach ($orphaned_meta_check_results[$orphaned_meta_check['meta_table']]['meta_keys'] as $key => $value) {

                        $orphaned_meta_check_results[$orphaned_meta_check['meta_table']]['meta_keys'][$key] = $value->meta_key;

                    }

                }

            }

            if ($orphaned_meta_check_results) {

?>
<h2><?php _e('Orphaned Meta Data'); ?></h2>
<p><?php _e('Meta data is information stored about an object (such as a comment, post, term or user). Sometimes the object is deleted and the meta data remains which can slow the database down.'); ?></p>
<ol>
<?php

                foreach ($orphaned_meta_check_results as $key => $orphaned_meta_check_result) {

                    echo '<li><strong>' . $orphaned_meta_check_result['total'] . ' orphaned meta items</strong> found in <strong>' . preg_replace('/^' . preg_quote($wpdb->prefix, '/') . '/', '', $key) . '</strong> table (<strong>Meta Keys:</strong> ' . esc_html(implode(', ', $orphaned_meta_check_result['meta_keys'])) . ')<br>
';

?>
<a href="<?php echo esc_url(wp_nonce_url(
                    add_query_arg(array(
                        'action' => 'wordpress_maintenance_download_orphaned_meta_data',
                        'table' => sanitize_key($key)
                    ), admin_url('admin-ajax.php')),
                    'download-orphaned-meta-data-' . sanitize_key($key)
        )); ?>" class="button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Download Orphaned Meta Data', 'wordpress-maintenance'); ?></a>
<span data-table="<?php echo sanitize_key($key); ?>" class="wordpress-maintenance-delete-orphaned-meta-data button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Delete Orphaned Meta Data', 'wordpress-maintenance'); ?></span>
<?php

                    echo '</li>
';

                }

?>
</ol>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-orphaned-meta-data').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete this orphaned meta data?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-orphaned-meta-data').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_orphaned_meta_data',
            	table: $(this).data('table'),
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-orphaned-meta-data'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

            $persistent_object_cache_issues = array();

    		$thresholds = apply_filters(
    			'site_status_persistent_object_cache_thresholds',
    			array(
    				'alloptions_count' => 500,
    				'alloptions_bytes' => 100000,
    				'comments_count'   => 1000,
    				'options_count'    => 1000,
    				'posts_count'      => 1000,
    				'terms_count'      => 1000,
    				'users_count'      => 1000,
    			)
    		);

    		$alloptions = wp_load_alloptions();
            $show_options_that_can_be_deleted = false;
            $show_post_types = false;

    		if ($thresholds['alloptions_count'] < count($alloptions)) {

                $show_options_that_can_be_deleted = true;
			    $persistent_object_cache_issues[] = sprintf(__('You have more than %s (%s) autoloaded options!'), number_format($thresholds['alloptions_count']), number_format(count($alloptions)));

    		}

    		if ($thresholds['alloptions_bytes'] < (strlen(serialize($alloptions)) - strlen(serialize(get_option('active_options')))) ) {

                $show_options_that_can_be_deleted = true;
			    $persistent_object_cache_issues[] = sprintf(__('You have more than %s (%s bytes) bytes of autoloaded options!'), number_format($thresholds['alloptions_bytes']), number_format((strlen(serialize($alloptions)) - strlen(serialize(get_option('active_options'))))));

    		}

    		$table_names = implode("','", array($wpdb->comments, $wpdb->options, $wpdb->posts, $wpdb->terms, $wpdb->users));

    		$results = $wpdb->get_results(
    			$wpdb->prepare(
    				"SELECT TABLE_NAME AS 'table', TABLE_ROWS AS 'rows', SUM(data_length + index_length) as 'bytes' FROM information_schema.TABLES WHERE TABLE_SCHEMA = %s AND TABLE_NAME IN ('$table_names') GROUP BY TABLE_NAME;",
    				DB_NAME
        			),
    			OBJECT_K
    		);

    		$threshold_map = array(
    			'comments_count' => $wpdb->comments,
    			'options_count'  => $wpdb->options,
    			'posts_count'    => $wpdb->posts,
    			'terms_count'    => $wpdb->terms,
    			'users_count'    => $wpdb->users,
    		);

    		foreach ($threshold_map as $threshold => $table) {

    			if ($thresholds[$threshold] <= $results[$table]->rows) {

    			    $persistent_object_cache_issues[] = sprintf(__('You have more than %s rows (%s) in the %s table in the database!'), number_format($thresholds[$threshold]), number_format($results[$table]->rows), str_replace($wpdb->prefix, '', $table));

                    if ('options_count' === $threshold) {

                        $show_options_that_can_be_deleted = true;

                    } elseif ('posts_count' === $threshold) {

                        $show_post_types = true;

                    }

    			}

    		}

            if ($show_options_that_can_be_deleted) {

                update_option('track_active_options', true);

            } else {

                delete_option('active_options');
                delete_option('track_active_options');

            }

            if ($show_options_that_can_be_deleted || ($persistent_object_cache_issues && !wp_using_ext_object_cache())) {

                if ($show_options_that_can_be_deleted && wp_using_ext_object_cache()) {

?>
<h2><?php _e('Transients and Options'); ?></h2>
<p><?php _e('Although you are using a Persistent Object Cache, it\'s still not good to have too many and / or too large auto-loading options in your wp_options table in the database ...'); ?></p>
<ol>
<?php

                    foreach ($persistent_object_cache_issues as $persistent_object_cache_issue) {

                        if (false !== strpos($persistent_object_cache_issue, 'option')) {

                            echo '<li>' . $persistent_object_cache_issue . '</li>';

                        }

                    }

?>
</ol>
<?php

                } else {

?>
<h2><?php _e('Persistent Object Cache'); ?></h2>
<p><?php _e('Since WordPress v6.1 the Site Health check has started recommending if you should be using a "Persistent Object Cache" without actually explaining why. Here\'s why ...'); ?></p>
<ol>
<?php

                    foreach ($persistent_object_cache_issues as $persistent_object_cache_issue) {

                        echo '<li>' . $persistent_object_cache_issue . '</li>';

                    }

?>
</ol>
<?php

                }

                $active_options = get_option('active_options');
                $active_options_details = array();

                if (!wp_using_ext_object_cache() && $show_post_types) {

?>
<h3><?php _e('Post Types:'); ?></h3>
<p><?php _e('To give you an idea of why you have so many rows in the wp_posts table, these are the types of posts that are in the database so you know why there are so many posts!'); ?></p>
<?php

                    $post_types = $wpdb->get_results("SELECT post_type, COUNT(post_type) as post_type_total FROM $wpdb->posts GROUP BY post_type ORDER BY post_type_total DESC;");

?>
<ul class="ul-disc">
<?php

                    foreach ($post_types as $post_type) {

?>
<li><?php echo esc_html($post_type->post_type) . ': ' . absint($post_type->post_type_total); ?></li>
<?php

                    }

?>
</ul>
<?php

                }

                if ($show_options_that_can_be_deleted && $active_options && is_array($active_options)) {

                    $allautoloadoptions = $alloptions;
        			$alloptions_db = $wpdb->get_results("SELECT option_name, option_value FROM $wpdb->options");
            		$alloptions = array();
                    $total_transients = 0;
                    $total_autoloaded_transients = array();
                    $autoloaded_transients = array();

            		foreach ((array) $alloptions_db as $o) {

                        if (
                            0 === substr_compare($o->option_name, '_transient_', 0, 11) ||
                            0 === substr_compare($o->option_name, '_site_transient_', 0, 16)
                        ) {

                            if (isset($allautoloadoptions[$o->option_name])) {

                                $total_autoloaded_transients[$o->option_name] = $o->option_value;

                            } else {

                                $total_transients++;

                            }

                        }

            			$alloptions[$o->option_name] = $o->option_value;

    	        	}

                    if ($total_transients || $total_autoloaded_transients) {

                        if ($total_autoloaded_transients) {

                            foreach ($total_autoloaded_transients as $option_name => $option_value) {

                                $autoloaded_transients[] = '<div class="wm_option_' . esc_attr($option_name) . '" title="' . esc_attr(print_r(maybe_unserialize($option_value), true)) . '">' . esc_html($option_name) . ' (' . number_format(strlen($option_value)/(1<<10), 2) . 'Kb' . ') <span style="color: red;">Autoloaded</span> <span class="dashicons dashicons-controls-repeat wm-deautoload-option" style="cursor: pointer;" title="Turn off autoloading of ' . esc_attr($option_name) . '" data-option="' . esc_attr($option_name) . '"></span> <span class="dashicons dashicons-trash wm-delete-option" style="cursor: pointer;" title="Delete transient ' . esc_attr($option_name) . '" data-option="' . esc_attr($option_name) . '"></span> <a href="https://www.google.com/search?q=' . urlencode('"' . $option_name . '" AND (wordpress OR plugin OR theme)') . '" target="_blank" style="text-decoration: none;"><span class="dashicons dashicons-search"></span></a></div>';

                            }

                        }

?>
<h3><?php _e('Transients:'); ?></h3>
<p><?php _e('Transients are stored in the wp_options table and are bits of temporary, cached data.'); ?></p>
<?php

                        if ($total_autoloaded_transients) {

?>
<p><span class="wordpress-maintenance-delete-autoloaded-transients button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>" style="color: red;"><?php _e('Delete Autoloaded Transients', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d autoloaded transient.', 'Delete %d autoloaded transients.', count($total_autoloaded_transients), 'wordpress-maintenance'), count($total_autoloaded_transients)); echo ' '; _e('Transients are meant to be a way for the site to store temporary information. These ones, however, have been added without an expiration date, so they\'ll never be deleted on their own!', 'wordpress-maintenance')?></p>
<?php

                        }

?>
<p><span class="wordpress-maintenance-delete-all-transients button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>" style="color: red;"><?php _e('Delete All Transients', 'wordpress-maintenance'); ?></span> <?php printf(_n('Delete %d transient.', 'Delete %d transients.', $total_transients + count($total_autoloaded_transients), 'wordpress-maintenance'), $total_transients + count($total_autoloaded_transients)); echo ' '; _e('Transients are a way for the site to store temporary information, even after deleting these, most will be re-added.', 'wordpress-maintenance')?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-delete-all-transients, .wordpress-maintenance-delete-autoloaded-transients').click(function() {
        var transient_type;
        if (this.classList.contains('wordpress-maintenance-delete-autoloaded-transients')) {
            transient_type = '<?php _e('autoloaded', 'wordpress-maintenance'); ?>';
        } else {
            transient_type = '<?php _e('all', 'wordpress-maintenance'); ?>';
        }
        if (confirm('<?php _e('Are you sure you want to delete ', 'wordpress-maintenance'); ?>' + transient_type + '<?php _e(' transients?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-delete-all-transients').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_delete_all_transients',
            	transient_type: transient_type,
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-all-transients'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

                    }

                    unset($alloptions['active_options']);
                    unset($alloptions['rewrite_rules']);

                    foreach ($active_options as $option_name) {

                        if (isset($alloptions[$option_name]) &&
                            0 !== substr_compare($option_name, '_transient_', 0, 11) &&
                            0 !== substr_compare($option_name, '_site_transient_', 0, 16)
                        ) {

                            $active_options_details[$option_name] = strlen($alloptions[$option_name]);

                        }

                        unset($alloptions[$option_name]);

                    }

                    arsort($active_options_details);
                    $largest_active_options = array_slice($active_options_details, 0, 10);
                    $largest_active_options_html = array();

                    foreach ($largest_active_options as $option_name => $option_size) {

                        $option_group = explode('_', $option_name);

                        if (isset($option_group[0])) {

                            $option_group = $option_group[0];

                        } else {

                            $option_group = false;

                        }

                        $largest_active_options_html[] = '<div class="wm_option_' . esc_attr($option_name) . '">' . esc_html($option_name) . ' (' . number_format($option_size/(1<<10), 2) . 'Kb' . ')' . (isset($allautoloadoptions[$option_name]) ? ' <span style="color: red;">Autoloaded</span> <span class="dashicons dashicons-controls-repeat wm-deautoload-option" style="cursor: pointer;" title="Turn off autoloading of ' . esc_attr($option_name) . '" data-option="' . esc_attr($option_name) . '"></span>' : '') . ' <span class="dashicons dashicons-trash wm-delete-option" style="cursor: pointer;" title="Delete option ' . esc_attr($option_name) . '" data-option="' . esc_attr($option_name) . '"></span> <a href="https://www.google.com/search?q=' . urlencode('"' . $option_name . '" AND (wordpress OR plugin OR theme)') . '" target="_blank" style="text-decoration: none;"><span class="dashicons dashicons-search"></span></a></div>';

                    }

                    $inactive_options = array();
                    ksort($alloptions);

                    foreach ($alloptions as $option_name => $option_value) {

                        $option_group = explode('_', $option_name);

                        if (isset($option_group[0])) {

                            $option_group = $option_group[0];

                        } else {

                            $option_group = false;

                        }

                        if (
                            0 !== substr_compare($option_name, '_transient_', 0, 11) &&
                            0 !== substr_compare($option_name, '_site_transient_', 0, 16)
                        ) {

                            $inactive_options[] = '<div class="wm_option_' . esc_attr($option_name) . '" title="' . esc_attr(print_r(maybe_unserialize($option_value), true)) . '">' . esc_html($option_name) . ' (' . number_format(strlen($option_value)/(1<<10), 2) . 'Kb' . ')' . (isset($allautoloadoptions[$option_name]) ? ' <span style="color: red;">Autoloaded</span> <span class="dashicons dashicons-controls-repeat wm-deautoload-option" style="cursor: pointer;" title="Turn off autoloading of ' . esc_attr($option_name) . '" data-option="' . esc_attr($option_name) . '"></span>' : '') . ' <span class="dashicons dashicons-trash wm-delete-option" style="cursor: pointer;" title="Delete option ' . esc_attr($option_name) . '" data-option="' . esc_attr($option_name) . '"></span>' . ($option_group ? ' <span class="dashicons dashicons-warning wm-delete-option-group" style="cursor: pointer;" title="Delete ALL inactive options starting with ' . esc_attr($option_group) . '_" data-option-group="' . esc_attr($option_group) . '"></span>' : '') . ' <a href="https://www.google.com/search?q=' . urlencode('"' . $option_name . '" AND (wordpress OR plugin OR theme)') . '" target="_blank" style="text-decoration: none;"><span class="dashicons dashicons-search"></span></a></div>';

                        }

                        $alloptions[$option_name] = strlen($option_value);

                    }

                    arsort($alloptions);
                    $largest_inactive_options = $alloptions;

                    foreach ($largest_inactive_options as $option_name => $option_size) {

                        if (
                            0 === substr_compare($option_name, '_transient_', 0, 11) ||
                            0 === substr_compare($option_name, '_site_transient_', 0, 16)
                        ) {

                            unset($largest_inactive_options[$option_name]);

                        }

                    }

                    $largest_inactive_options = array_slice($largest_inactive_options, 0, 10);
                    $largest_options = array();

                    foreach ($largest_inactive_options as $option_name => $option_size) {

                        $option_group = explode('_', $option_name);

                        if (isset($option_group[0])) {

                            $option_group = $option_group[0];

                        } else {

                            $option_group = false;

                        }

                        $largest_options[] = '<div class="wm_option_' . esc_attr($option_name) . '">' . esc_html($option_name) . ' (' . number_format($option_size/(1<<10), 2) . 'Kb' . ')' . (isset($allautoloadoptions[$option_name]) ? ' <span style="color: red;">Autoloaded</span> <span class="dashicons dashicons-controls-repeat wm-deautoload-option" style="cursor: pointer;" title="Turn off autoloading of ' . esc_attr($option_name) . '" data-option="' . esc_attr($option_name) . '"></span>' : '') . ' <span class="dashicons dashicons-trash wm-delete-option" style="cursor: pointer;" title="Delete option ' . esc_attr($option_name) . '" data-option="' . esc_attr($option_name) . '"></span>' . ($option_group ? ' <span class="dashicons dashicons-warning wm-delete-option-group" style="cursor: pointer;" title="Delete ALL inactive options starting with ' . esc_attr($option_group) . '_" data-option-group="' . esc_attr($option_group) . '"></span>' : '') . ' <a href="https://www.google.com/search?q=' . urlencode('"' . $option_name . '" AND (wordpress OR plugin OR theme)') . '" target="_blank" style="text-decoration: none;"><span class="dashicons dashicons-search"></span></a></div>';

                    }

?>
<h3><?php _e('Options:'); ?></h3>
<p><?php _e('We\'ve developed a bunch of tools to help you further understand and manage the options (and autoloaded transients) in the wp_options table in the database.'); ?></p>
<ul>
    <li><span class="dashicons dashicons-controls-repeat"></span> - <?php _e('turn off autoloading for a single option or transient'); ?></li>
    <li><span class="dashicons dashicons-trash"></span> - <?php _e('delete a single option or transient'); ?></li>
    <li><span class="dashicons dashicons-warning"></span> - <?php _e('delete a group of inactive options'); ?></li>
    <li><span class="dashicons dashicons-search"></span> - <?php _e('search for more information on a particular option or transient'); ?></li>
</ul>
<h4><?php _e('Autoloaded transients:'); ?></h4>
<p><?php _e('These transients are autoloaded because their expiration date has not been set!'); ?></p>
<p><?php echo implode(PHP_EOL, $autoloaded_transients); ?></p>
<h4><?php _e('Largest inactive options:'); ?></h4>
<p><?php _e('Just because these options have not recently been used does not mean that they are not important so be careful here!'); ?></p>
<p><?php echo implode(PHP_EOL, $largest_options); ?></p>
<h4><?php _e('Largest active options:'); ?></h4>
<p><?php _e('These options are definitely being used so be extra careful here!'); ?></p>
<p><?php echo implode(PHP_EOL, $largest_active_options_html); ?></p>
<h4><?php _e('All options that have not recently been used:'); ?></h4>
<p><?php _e('Just because these options have not recently been used does not mean that they are not important so be careful here!'); ?></p>
<p><?php echo implode(PHP_EOL, $inactive_options); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wm-deautoload-option').click(function() {
        if (confirm('<?php _e('Are you sure you want to turn off autoloading for', 'wordpress-maintenance'); ?> "' + this.getAttribute('data-option') + '"')) {
            $(this).unbind('click')
            var targetOption = this,
            data = {
            	action: 'wordpress_maintenance_deautoload_option',
            	option: this.getAttribute('data-option'),
            	_ajax_nonce: '<?php echo wp_create_nonce('deautoload-option'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
    $('.wm-delete-option').click(function() {
        if (confirm('<?php _e('Are you sure you want to delete the option', 'wordpress-maintenance'); ?> "' + this.getAttribute('data-option') + '"')) {
            $(this).unbind('click')
            var targetOption = this,
            data = {
            	action: 'wordpress_maintenance_delete_option',
            	option: this.getAttribute('data-option'),
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-option'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        targetOption.parentElement.remove();
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
    $('.wm-delete-option-group').click(function() {
        if (confirm('<?php _e('WARNING! Are you SURE you want to delete ALL inactive options starting with ', 'wordpress-maintenance'); ?> "' + this.getAttribute('data-option-group') + '_? This CANNOT be undone!"')) {
            $(this).unbind('click')
            data = {
            	action: 'wordpress_maintenance_delete_option_group',
            	option_group: this.getAttribute('data-option-group'),
            	_ajax_nonce: '<?php echo wp_create_nonce('delete-option-group'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

                }


            }

?>

</div>
<div class="wrap tab_content settings"<?php echo ('settings' === $current_tab ? '' : ' style="display: none;"'); ?>>
<h2><?php _e('Settings', 'template-editor'); ?></h2>
<?php

            $can_edit_wp_config = true;

			if (!file_exists(wmCommon::get_home_path() . 'wp-config.php')) {

                $can_edit_wp_config = false;

?>
<p style="color: red;"><strong>Cannot find "wp-config.php"!</strong></p>
<?php

            } elseif (!is_writable(wmCommon::get_home_path() . 'wp-config.php')) {

                $can_edit_wp_config = false;

?>
<p style="color: red;"><strong>Cannot write to "wp-config.php"!</strong></p>
<?php

            }

            if ($can_edit_wp_config) {

				$wp_config = explode("\n", implode('', file(wmCommon::get_home_path() . 'wp-config.php')));
				$toggle_debug_log = __('Enable', 'wordpress-maintenance');
				$toggle_wp_cron = __('Disable', 'wordpress-maintenance');
				$toggle_file_edit = __('Disable', 'wordpress-maintenance');

				foreach ($wp_config as $line) {

					if (preg_match('/^define\(\s?["\']{1}WP_DEBUG["\']{1},\s?true/i', $line)) {

					    $toggle_debug_log = __('Disable', 'wordpress-maintenance');

					}

					if (preg_match('/^define\(\s?["\']{1}DISABLE_WP_CRON["\']{1},\s?true/i', $line)) {

					    $toggle_wp_cron = __('Enable', 'wordpress-maintenance');

					}

					if (preg_match('/^define\(\s?["\']{1}DISALLOW_FILE_EDIT["\']{1},\s?true/i', $line)) {

					    $toggle_file_edit = __('Enable', 'wordpress-maintenance');

					}
				}

?>
<p><span class="wordpress-maintenance-toggle-debug-log button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php echo $toggle_debug_log; ?> debug.log</span> <?php echo $toggle_debug_log; ?> debug.log <?php _e('in wp-config.php.', 'wordpress-maintenance'); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-toggle-debug-log').click(function() {
        if (confirm('<?php printf(__('Are you sure you want to %s debug.log?', 'wordpress-maintenance'), strtolower($toggle_debug_log)); ?>')) {
            $('.wordpress-maintenance-toggle-debug-log').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_toggle_debug_log',
            	_ajax_nonce: '<?php echo wp_create_nonce('toggle-debug-log'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>

<p><span class="wordpress-maintenance-toggle-wp-cron button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php echo $toggle_wp_cron; ?> WP CRON</span> <?php echo $toggle_wp_cron; ?> WP CRON <?php _e('in wp-config.php.', 'wordpress-maintenance'); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-toggle-wp-cron').click(function() {
        if (confirm('<?php printf(__('Are you sure you want to %s WP CRON?', 'wordpress-maintenance'), strtolower($toggle_wp_cron)); ?>')) {
            $('.wordpress-maintenance-toggle-wp-cron').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_toggle_wp_cron',
            	_ajax_nonce: '<?php echo wp_create_nonce('toggle-wp-cron'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>

<p><span class="wordpress-maintenance-toggle-file-edit button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php echo $toggle_file_edit; ?> File Edit</span> <?php echo $toggle_file_edit; ?> <?php _e('plugin and theme editor in wp-config.php.', 'wordpress-maintenance'); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-toggle-file-edit').click(function() {
        if (confirm('<?php printf(__('Are you sure you want to %s file edit?', 'wordpress-maintenance'), strtolower($toggle_file_edit)); ?>')) {
            $('.wordpress-maintenance-toggle-file-edit').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_toggle_file_edit',
            	_ajax_nonce: '<?php echo wp_create_nonce('toggle-file-edit'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>

<p><span class="wordpress-maintenance-<?php echo (((defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED)) ? 'enable' : 'disable'); ?>-updates button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php echo (((defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED)) ? __('Enable Updates', 'wordpress-maintenance') : __('Disable Updates', 'wordpress-maintenance')); ?></span> <?php echo (((defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED)) ? __('Enable automatic core updates.', 'wordpress-maintenance') : __('Disable automatic core updates.', 'wordpress-maintenance')); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-<?php echo (((defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED)) ? 'enable' : 'disable'); ?>-updates').click(function() {
        if (confirm('<?php echo (((defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED)) ? 'Are you sure you want to enable core updates?' : 'Are you sure you want to disable core updates?'); ?>')) {
            $('.wordpress-maintenance-<?php echo (((defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED)) ? 'enable' : 'disable'); ?>-updates').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_<?php echo (((defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED)) ? 'enable' : 'disable'); ?>_updates',
            	_ajax_nonce: '<?php echo wp_create_nonce((((defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED)) ? 'enable' : 'disable') . '-updates'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<?php

            }

?>
<p><span class="wordpress-maintenance-<?php echo (get_option('webd_uk_service_notifications') ? 'disable' : 'enable'); ?>-service-notifications button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php echo (get_option('webd_uk_service_notifications') ? __('Disable Service Email Notifications', 'wordpress-maintenance') : __('Enable Service Email Notifications', 'wordpress-maintenance')); ?></span> <?php echo (get_option('webd_uk_service_notifications') ? __('Disable service email notifications.', 'wordpress-maintenance') : __('Enable service email notifications.', 'wordpress-maintenance')); ?></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-<?php echo (get_option('webd_uk_service_notifications') ? 'disable' : 'enable'); ?>-service-notifications').click(function() {
        if (confirm('<?php echo (get_option('webd_uk_service_notifications') ? 'Are you sure you want to disable service notifications?' : 'Are you sure you want to enable service notifications?'); ?>')) {
            $('.wordpress-maintenance-<?php echo (get_option('webd_uk_service_notifications') ? 'disable' : 'enable'); ?>-service-notifications').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_<?php echo (get_option('webd_uk_service_notifications') ? 'disable' : 'enable'); ?>_service_notifications',
            	_ajax_nonce: '<?php echo wp_create_nonce((get_option('webd_uk_service_notifications') ? 'disable' : 'enable') . '-service-notifications'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php } ?>
<p>Daily delete all but the latest <input type="number" id="wordpress-maintenance-revision-retention" value="<?php echo absint(get_option('wm_revision_retention')); ?>" /> revisions (a value of 0 will disable this feature)<br />
<span class="wordpress-maintenance-set-revision-retention button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Set Revision Retention Value', 'wordpress-maintenance'); ?></span></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-set-revision-retention').click(function() {
        if (confirm('<?php _e('Are you sure you want to set to delete revisions daily?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-set-revision-retention').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_set_revision_retention',
            	_ajax_nonce: '<?php echo wp_create_nonce('set-revision-retention'); ?>',
            	revisions: $('#wordpress-maintenance-revision-retention').val()
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php }

?>
<h3>Cloudflare Auto Protect</h3>
<p>If enabled, this feature will enable "Under Attack Mode" on Cloudflare automatically when your server's CPUs are overloaded. It checks the status of the server every fifteen minutes and enables or disables "Under Attack Mode" in your Cloudflare dashboard accordingly.</p>
<p>To enable this feature, sign into your Cloudflare account and create an API token as follows:</p>
<ol>
<li>Go to: Cloudflare - My Profile - API Tokens - Create Custom Token</li>
<li>Name the token: WordPress Maintenance</li>
<li>Set "Permissions": Zone - Zone Settings - Edit</li>
<li>Set "Permissions": Zone - Firewall Services - Edit</li></li>
<li>Click "Continue to summary" and then "Create Token"</li>
</ol>
<p>If you use your server to fire WP cron, don't forget to go to "Security - WAF - Tools" and add a rule to allow your server's IP address.</p>
<p>Cloudflare API Token: <input type="text" id="wordpress-maintenance-cloudflare-api-token" value="<?php echo esc_attr(get_option('wm_cloudflare_api_token')); ?>" /> (an empty value will disable this feature)<br />
<span class="wordpress-maintenance-set-cloudflare-api-token button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>"><?php _e('Set Cloudflare API Token', 'wordpress-maintenance'); ?></span></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-set-cloudflare-api-token').click(function() {
        if (confirm('<?php _e('Are you sure you want to enable Cloudflare auto protect?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-set-cloudflare-api-token').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_set_cloudflare_api_token',
            	_ajax_nonce: '<?php echo wp_create_nonce('set-cloudflare-api-token'); ?>',
            	cloudflare_api_token: $('#wordpress-maintenance-cloudflare-api-token').val()
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    } else {
                        alert('<?php _e('1Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo add_query_arg('page', 'wordpress-maintenance', admin_url('tools.php')); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php }

?>
</div>
<div class="wrap tab_content phpinfo"<?php echo ('phpinfo' === $current_tab ? '' : ' style="display: none;"'); ?>>
<h2><?php _e('PHP Info', 'template-editor'); ?></h2>
<?php

    ob_start();
    phpinfo();
    $phpinfo = ob_get_contents();
    ob_end_clean();
    $phpinfo = preg_replace('%^.*<body>(.*)</body>.*$%ms', '$1', $phpinfo);
    echo "
        <style type='text/css'>
            #phpinfo {}
            #phpinfo pre {margin: 0; font-family: monospace;}
            #phpinfo a:link {color: #009; text-decoration: none; background-color: #fff;}
            #phpinfo a:hover {text-decoration: underline;}
            #phpinfo table {border-collapse: collapse; border: 0; width: 934px; box-shadow: 1px 2px 3px #ccc;}
            #phpinfo .center {text-align: center;}
            #phpinfo .center table {margin: 1em auto; text-align: left;}
            #phpinfo .center th {text-align: center !important;}
            #phpinfo td, th {border: 1px solid #666; font-size: 75%; vertical-align: baseline; padding: 4px 5px;}
            #phpinfo h1 {font-size: 150%;}
            #phpinfo h2 {font-size: 125%;}
            #phpinfo .p {text-align: left;}
            #phpinfo .e {background-color: #ccf; width: 300px; font-weight: bold;}
            #phpinfo .h {background-color: #99c; font-weight: bold;}
            #phpinfo .v {background-color: #ddd; max-width: 300px; overflow-x: auto; word-wrap: break-word;}
            #phpinfo .v i {color: #999;}
            #phpinfo img {float: right; border: 0;}
            #phpinfo hr {width: 934px; background-color: #ccc; border: 0; height: 1px;}
        </style>
        <div id='phpinfo'>
            $phpinfo
        </div>
";

?>
</div>
</div>
<?php

        }

        public function wordpress_maintenance_toggle_debug_log() {

            check_ajax_referer('toggle-debug-log');

            if ($this->request_permission() && current_user_can('manage_options')) {

				$wp_config = explode("\n", implode('', file(wmCommon::get_home_path() . 'wp-config.php')));
				$newdata = '';
				$found = false;

                if (defined('WP_DEBUG') && WP_DEBUG) {

                    if (file_exists(WP_CONTENT_DIR . '/debug.log')) {

                        add_action('shutdown', array($this, 'wordpress_maintenance_shutdown'));

                    }

                    $wp_debug = 'define(\'WP_DEBUG\', false);
';

                } else {

                    $wp_debug = 'define(\'WP_DEBUG\', true);
define(\'WP_DEBUG_DISPLAY\', false);
define(\'WP_DEBUG_LOG\', true);
';

                }

				foreach ($wp_config as $line) {

					if (preg_match('/^define\(\s?["\']{1}WP_DEBUG/i', $line)) {

                        if (!$found) {

                            $newdata .= $wp_debug;
						    $found = true;

                        }

					} else {

						$newdata .= "$line\n";

					}

				}

				$f = @fopen(wmCommon::get_home_path() . 'wp-config.php', 'w');
				fwrite($f, rtrim($newdata));

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_toggle_wp_cron() {

            check_ajax_referer('toggle-wp-cron');

            if ($this->request_permission() && current_user_can('manage_options')) {

				$wp_config = explode("\n", implode('', file(wmCommon::get_home_path() . 'wp-config.php')));
				$newdata = '';
				$found = false;

                if (defined('DISABLE_WP_CRON') && DISABLE_WP_CRON) {

                    $wp_cron = 'define(\'DISABLE_WP_CRON\', false);
';

                } else {

                    $wp_cron = 'define(\'DISABLE_WP_CRON\', true);
';

                }

				foreach ($wp_config as $line) {

					if (preg_match('/^define\(\s?["\']{1}DISABLE_WP_CRON/i', $line)) {

                        if (!$found) {

                            $newdata .= $wp_cron;
						    $found = true;

                        }

					} else {

						$newdata .= "$line\n";

					}

				}

				if (!$found) {

					$newdata = '';

					foreach ($wp_config as $line) {

						if (preg_match('/^\$table_prefix/', $line)) {

	                        if (!$found) {

	                            $newdata .= "$line\n\n" . $wp_cron . "\n";
							    $found = true;

	                        }

						} else {

							$newdata .= "$line\n";

						}

					}

				}

				$f = @fopen(wmCommon::get_home_path() . 'wp-config.php', 'w');
				fwrite($f, rtrim($newdata));

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_toggle_file_edit() {

            check_ajax_referer('toggle-file-edit');

            if ($this->request_permission() && current_user_can('manage_options')) {

				$wp_config = explode("\n", implode('', file(wmCommon::get_home_path() . 'wp-config.php')));
				$newdata = '';
				$found = false;

                if (defined('DISALLOW_FILE_EDIT') && DISALLOW_FILE_EDIT) {

                    $wp_cron = 'define(\'DISALLOW_FILE_EDIT\', false);
';

                } else {

                    $wp_cron = 'define(\'DISALLOW_FILE_EDIT\', true);
';

                }

				foreach ($wp_config as $line) {

					if (preg_match('/^define\(\s?["\']{1}DISALLOW_FILE_EDIT/i', $line)) {

                        if (!$found) {

                            $newdata .= $wp_cron;
						    $found = true;

                        }

					} else {

						$newdata .= "$line\n";

					}

				}

				if (!$found) {

					$newdata = '';

					foreach ($wp_config as $line) {

						if (preg_match('/^\$table_prefix/', $line)) {

	                        if (!$found) {

	                            $newdata .= "$line\n\n" . $wp_cron . "\n";
							    $found = true;

	                        }

						} else {

							$newdata .= "$line\n";

						}

					}

				}

				$f = @fopen(wmCommon::get_home_path() . 'wp-config.php', 'w');
				fwrite($f, rtrim($newdata));

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_enable_updates() {

            check_ajax_referer('enable-updates');

            if ($this->request_permission() && current_user_can('manage_options')) {

				$wp_config = explode("\n", implode('', file(wmCommon::get_home_path() . 'wp-config.php')));
				$newdata = '';

				foreach ($wp_config as $line) {

					if (!preg_match('/^define\(["\']{1}AUTOMATIC_UPDATER_DISABLED["\']{1}, true\);/i', $line)) {

						$newdata .= "$line\n";

					}

				}

				$f = @fopen(wmCommon::get_home_path() . 'wp-config.php', 'w');
				fwrite($f, rtrim($newdata));

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_disable_updates() {

            check_ajax_referer('disable-updates');

            if ($this->request_permission() && current_user_can('manage_options')) {

				$wp_config = explode("\n", implode('', file(wmCommon::get_home_path() . 'wp-config.php')));
				$newdata = '';
				$found = false;

                if (defined('AUTOMATIC_UPDATER_DISABLED') && AUTOMATIC_UPDATER_DISABLED) {

                    wp_send_json_error();

                } else {

                    $wp_cron = 'define(\'AUTOMATIC_UPDATER_DISABLED\', true);
';

                }

				foreach ($wp_config as $line) {

					if (preg_match('/^define\(\s?["\']{1}AUTOMATIC_UPDATER_DISABLED/i', $line)) {

                        if (!$found) {

                            $newdata .= $wp_cron;
						    $found = true;

                        }

					} else {

						$newdata .= "$line\n";

					}

				}

				if (!$found) {

					$newdata = '';

					foreach ($wp_config as $line) {

						if (preg_match('/^\$table_prefix/', $line)) {

	                        if (!$found) {

	                            $newdata .= "$line\n\n" . $wp_cron . "\n";
							    $found = true;

	                        }

						} else {

							$newdata .= "$line\n";

						}

					}

				}

				$f = @fopen(wmCommon::get_home_path() . 'wp-config.php', 'w');
				fwrite($f, rtrim($newdata));

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_enable_service_notifications() {

            check_ajax_referer('enable-service-notifications');

            if ($this->request_permission() && current_user_can('manage_options')) {

                update_option('webd_uk_service_notifications', true, false);

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_disable_service_notifications() {

            check_ajax_referer('disable-service-notifications');

            if ($this->request_permission() && current_user_can('manage_options')) {

                delete_option('webd_uk_service_notifications');

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_disable_links() {

            check_ajax_referer('disable-links');

            if ($this->request_permission() && current_user_can('manage_options')) {

                global $wpdb;

                $wpdb->query("TRUNCATE TABLE {$wpdb->prefix}links");
                update_option('link_manager_enabled', 0);

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

		public function wordpress_maintenance_delete_upload_path_option() {

            check_ajax_referer('delete-upload-path-option');

            if ($this->request_permission() && current_user_can('manage_options')) {

				delete_option('upload_path');

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

		}

        public function wordpress_maintenance_convert_tables_to_innodb() {

            check_ajax_referer('convert-tables-to-innodb');

            if ($this->request_permission() && current_user_can('manage_options')) {

                global $wpdb;
                $database_tables = $wpdb->get_results("SHOW TABLE STATUS");

                foreach ($database_tables as $key => $database_table) {

                    if ('InnoDB' !== $database_table->Engine) {

                        $wpdb->query("ALTER TABLE {$database_table->Name} ENGINE=INNODB");

                    }

                }

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_delete_empty_wp_cron_events() {

            check_ajax_referer('delete-empty-wp-cron-events');

            if ($this->request_permission() && current_user_can('manage_options')) {

                $empty_event_hooks = $this->wordpress_maintenance_get_empty_cron_event_hooks();

                foreach ($empty_event_hooks as $hook) {

                    wp_clear_scheduled_hook($hook);

                }

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_delete_inactive_widgets() {

            check_ajax_referer('delete-inactive-widgets');

            if ($this->request_permission() && current_user_can('manage_options')) {

                $inactive_widgets = wp_get_sidebars_widgets();
                $active_widgets = array();

                if (is_array($inactive_widgets) && $inactive_widgets) {

                    foreach ($inactive_widgets as $sidebar_id => $widgets) {

                        if ('wp_inactive_widgets' !== $sidebar_id && (is_registered_sidebar($sidebar_id) || !$widgets)) {

                    	    $active_widgets[$sidebar_id] = $widgets;

                    	} else {

                    		foreach ( $widgets as $key => $widget_id ) {

                    			$pieces = explode('-', $widget_id);
                			    $multi_number = array_pop($pieces);
                    			$id_base = implode('-', $pieces);
                    			$widget = get_option('widget_' . $id_base);
			                    unset($widget[$multi_number]);
                    			update_option('widget_' . $id_base, $widget);

                    		}

                    	}

                    }

            		wp_set_sidebars_widgets( $active_widgets );

                }

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }


        public function wordpress_maintenance_delete_languages() {

            if ($this->request_permission() && current_user_can('manage_options') && isset($_POST['file_type']) && in_array($_POST['file_type'], array('locales', 'plugins', 'themes'), true)) {

                $file_type = sanitize_key($_POST['file_type']);

                check_ajax_referer('delete-languages-' . $file_type);

                if (is_dir(WP_CONTENT_DIR . '/languages/')) {

        		    $iterator = new RecursiveDirectoryIterator(WP_CONTENT_DIR . '/languages/');

            		foreach (new RecursiveIteratorIterator($iterator, RecursiveIteratorIterator::CHILD_FIRST) as $file) {

            			$file_info = pathinfo($file->getPathname());

            			if ($file->isFile() && isset($file_info['extension']) && in_array(strtolower($file_info['extension']), array('mo', 'po', 'json'))) {

                            $language = $file_info['filename'];

                            if (str_contains($language, '-')) {

                                if (in_array($file_info['extension'], array('mo', 'po'))) {

                                    $directory = rtrim($file_info['dirname'],'/');
                                    $directory = substr($directory, strrpos($directory, '/') + 1);

                                    if ('languages' !== $directory) {

                                        $theme_or_plugin = substr($language, 0, strrpos($language, '-'));

                                        if (!is_dir(WP_CONTENT_DIR . '/' . $directory . '/' . $theme_or_plugin)) {

                                            if ($file_type === $directory) {

                                                unlink($file->getPathname());

                                            }

                                        }

                                    }

                                    $language = substr($language, strrpos($language, '-') + 1);

                                } else {

                                    $language = preg_replace('/-wc-admin-app$/', '-woocommerce', $language);
                                    $language = substr($language, 0, strrpos($language, '-'));

                                    if (str_contains($language, '-')) {

                                        $language = substr($language, strrpos($language, '-') + 1);

                                    }

                                }

                            }

                            if ($file_type === 'locales' && $language !== get_locale()) {

                                unlink($file->getPathname());

                            }

            			}

            		}

                }

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_delete_posts() {

            if ($this->request_permission() && current_user_can('manage_options') && isset($_POST['post_type']) && in_array($_POST['post_type'], array('flamingo_contact', 'flamingo_inbound', 'revision', 'any', 'oembed_cache'), true)) {

                $post_type = sanitize_key($_POST['post_type']);

                check_ajax_referer('delete-posts-' . $post_type);

                $args = array(
                    'post_type' => $post_type,
                    'numberposts' => -1,
                    'fields' => 'ids'
                );

                if ('revision' === $post_type) {

                    $args['post_status'] = 'inherit';

                } elseif ('any' === $post_type) {

                    $args['post_status'] = 'auto-draft';

                } elseif (isset($_POST['post_status']) && sanitize_key($_POST['post_status'])) {

                    $args['post_status'] = sanitize_key($_POST['post_status']);

                }

                $posts = get_posts($args);

                foreach ($posts as $post) {

                    wp_delete_post($post, true);

                }

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_delete_comments() {

            if ($this->request_permission() && current_user_can('manage_options') && isset($_POST['status'])) {

                $comment_status = sanitize_key($_POST['status']);

                check_ajax_referer('delete-' . (in_array($comment_status, array('hold', 'spam', 'trash', 'all'), true) ? $comment_status : '') . '-comments');

                $args = array(
                    'status' => $comment_status,
                );

                if ('all' === $comment_status) {

                    $args['type__not_in'] = array('order_note', 'review');

                }

                $comments = get_comments($args);

                foreach ($comments as $comment) {

                    wp_delete_comment($comment, true);

                }

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_close_comments() {

            check_ajax_referer('close-comments');

            if ($this->request_permission() && current_user_can('manage_options')) {

                global $wpdb;

                $wpdb->update( 
                    $wpdb->posts, 
                    array( 
                        'comment_status' => 'closed'
                    ),
                    array(
                        'comment_status' => 'open'
                    ),
                    array( 
                        '%s'
                    ),
                    array(
                        '%s'
                    )
                );

                update_option('default_comment_status', 'closed');

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_close_pings() {

            check_ajax_referer('close-pings');

            if ($this->request_permission() && current_user_can('manage_options')) {

                global $wpdb;

                $wpdb->update( 
                    $wpdb->posts, 
                    array( 
                        'ping_status' => 'closed'
                    ),
                    array(
                        'ping_status' => 'open'
                    ),
                    array( 
                        '%s'
                    ),
                    array(
                        '%s'
                    )
                );

                update_option('default_ping_status', 'closed');

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_change_admin_email() {

            check_ajax_referer('change-admin-email');

            if ($this->request_permission() && current_user_can('manage_options') && isset($_POST['email']) && sanitize_email($_POST['email'])) {

                update_option('admin_email', sanitize_email($_POST['email']));

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_set_revision_retention() {

            check_ajax_referer('set-revision-retention');

            if ($this->request_permission() && current_user_can('manage_options') && isset($_POST['revisions']) && absint($_POST['revisions'])) {

                update_option('wm_revision_retention', absint($_POST['revisions']));

                wp_send_json_success();

            } elseif($this->request_permission() && current_user_can('manage_options') && isset($_POST['revisions'])) {

                delete_option('wm_revision_retention');

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_recover_custom_css() {

            check_ajax_referer('recover-custom-css');

            if ($this->request_permission() && current_user_can('manage_options')) {

                $css_found = false;

                foreach (get_posts(array(
    'post_type' => 'custom_css',
    'numberposts' => -1
)) as $custom_css) {

                    if (get_stylesheet() === $custom_css->post_title) {

                        $css_found = true;
                        set_theme_mod('custom_css_post_id', $custom_css->ID);

                    }

                }

                if ($css_found) {

                    wp_send_json_success();

                } else {

                    wp_send_json_error();

                }

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_deautoload_option() {

            check_ajax_referer('deautoload-option');

            if ($this->request_permission() && current_user_can('manage_options') && isset($_POST['option']) && sanitize_key($_POST['option'])) {

                global $wpdb;

                $wpdb->update($wpdb->options, array('autoload' => 'no'), array('option_name' => sanitize_key($_POST['option'])));

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_delete_option() {

            check_ajax_referer('delete-option');

            if ($this->request_permission() && current_user_can('manage_options') && isset($_POST['option']) && sanitize_key($_POST['option'])) {

                if (delete_option($_POST['option'])) {

                    wp_send_json_success();

                } else {

                    wp_send_json_error();

                }

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_delete_option_group() {

            check_ajax_referer('delete-option-group');

            if ($this->request_permission() && current_user_can('manage_options') && isset($_POST['option_group']) && sanitize_key($_POST['option_group'])) {

                global $wpdb;

                $option_group = sanitize_key($_POST['option_group']) . '_';
                $active_options = get_option('active_options');

                if ($active_options && is_array($active_options)) {

        			$alloptions_db = $wpdb->get_results("SELECT option_name FROM $wpdb->options");
            		$alloptions = array();

            		foreach ((array) $alloptions_db as $o) {

                        if (!in_array($o->option_name, $active_options, true) && 0 === substr_compare($o->option_name, $option_group, 0, strlen($option_group))) {

                            delete_option($o->option_name);

                        }

    	        	}

                    wp_send_json_success();

                } else {

                    wp_send_json_error();

                }

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        private function wordpress_maintenance_get_empty_cron_event_hooks() {

            $cron_events  = _get_cron_array();
            $empty_event_hooks = array();

            if (is_array($cron_events)) {

                global $wp_filter;

                foreach ($cron_events as $cron_event) {

                    foreach ($cron_event as $hook => $value) {

                        if ('wc_admin_unsnooze_admin_notes' !== $hook &&!isset($wp_filter[$hook])) {

                            $empty_event_hooks[] = $hook;

                    	}

                    }

				}

			}

            return array_unique($empty_event_hooks);

        }

        private function request_permission() {

            $current_user = wp_get_current_user();

            return (isset($current_user->user_email) && (wmPremium::request_permission() || 'b55a52071eeb777fd8db89075431f1b7' === md5($current_user->user_email)));

        }

        public function wordpress_maintenance_shutdown() {

            unlink(WP_CONTENT_DIR . '/debug.log');

        }

        public function wordpress_maintenance_daily_schedule() {

            $revision_retention = absint(get_option('wm_revision_retention'));

            if ($revision_retention) {

                $revisions = get_posts(array(
                    'post_type' => 'revision',
                    'numberposts' => -1,
                    'fields' => 'ids',
                    'post_status' => 'inherit',
                    'orderby' => 'date',
                    'order'   => 'DESC'
                ));

                if ($revisions) {

                    array_splice($revisions, 0, $revision_retention);

                    if ($revisions) {

                        foreach ($revisions as $revision) {

                            wp_delete_post($revision, true);

                        }

                    }

                }

            }

        }

        public static function wp_mail_from_name($from_name) {

            return 'webd.uk';

        }

        public function wordpress_maintenance_hourly_schedule() {

            if (wp_using_ext_object_cache() && method_exists('Memcache','getStats')) {

                $memcache_obj = new Memcache;
                $memcache_obj->addServer('127.0.0.1', '11211');
                $memcache_stats = $memcache_obj->getStats();

                if ($memcache_stats) {

                    if ($memcache_stats['bytes'] / $memcache_stats['limit_maxbytes'] > 0.90) {

                        error_log('Memcached was flushed: ' . absint($memcache_stats['bytes'] / $memcache_stats['limit_maxbytes'] * 100) . '% used (' . self::readable_size($memcache_stats['bytes']) . ' of ' . self::readable_size($memcache_stats['limit_maxbytes']) . ')');
                        $memcache_obj->flush();

                    }

                }

            }

            if (
                get_option('webd_uk_service_notifications') &&
                absint(date('G')) > 9 &&
                absint(date('G')) < 16
            ) {

                require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

                // Force reload webd.uk available plugin updates
            	$plugin_updates = get_site_transient('update_plugins');
                set_site_transient('update_plugins', $plugin_updates);

                $plugin_updates = get_plugin_updates();
                $theme_updates = get_theme_updates();
                $core_updates = get_core_updates();
                $core_updates_final = array();
            	$check_php = wp_check_php_version();

                if (is_array($core_updates)) {

                    foreach ($core_updates as $update) {

                        if ('upgrade' === $update->response && 0 === count($core_updates_final)) {

                            $core_updates_final[] = $update;

                		}

                	}

                }

                $total_updates = count($core_updates_final) + count($plugin_updates) + count($theme_updates);

                if (
                    (
                        $total_updates > 9 ||
                        (isset($check_php['is_acceptable']) && !$check_php['is_acceptable']) ||
                        count($core_updates_final) ||
                        (
                            absint(get_option('webd_uk_service_last_notified')) &&
                            absint(get_option('webd_uk_service_last_notified')) < strtotime('-1 year')
                        )
                    ) &&
                    (
                        (
                            absint(get_option('webd_uk_service_last_notified')) &&
                            absint(get_option('webd_uk_service_last_notified')) < strtotime('-1 month')
                        ) ||
                        !absint(get_option('webd_uk_service_last_notified'))
                    )
                ) {

                    $message = 'Hello,

';

                    if ($total_updates) {

                        $message .= 'Your WordPress website has ' . $total_updates . ' ' . _n('update', 'updates', $total_updates, 'wordpress-maintenance') . ' waiting to be applied';

                        if (count($core_updates_final)) {

                            $message .= ' including an important core update';

                        }

                        if (isset($check_php['is_acceptable']) && !$check_php['is_acceptable']) {

                            $message .= ' and your site is running on an outdated version of PHP';

                        }

                        $message .= '!

';

                    } elseif (
                        isset($check_php['is_acceptable']) &&
                        !$check_php['is_acceptable']
                    ) {

                        $message .= 'Your WordPress website hasn\'t had a service for a while and your site is running on an outdated version of PHP!

';

                    } else {

                        $message .= 'Your WordPress website hasn\'t had a service for a while and we recommend doing so at least once per year to make sure there are no major issues going undetected. You can run a quick scan on your site here https://webd.uk/scan/

';

                    }

                    $message .= 'If you would like us to run a full service on your site please reply to this email.

Kind regards,

Oliver



---

Webd Ltd
https://webd.uk

CONFIDENTIALITY NOTICE
This communication and the information it contains is intended for the person or organisation to whom it is addressed. Its contents are confidential and may be protected in law. Unauthorised use, copying or disclosure of any of it may be unlawful. If you are not the intended recipient, please contact us immediately.

Company Number: 12518645
Registered Office: 26 High Street, Battle, East Sussex TN33 0EA, UK
';

                    $headers = array(
                        'Reply-To: webd.uk <info@webd.uk>'
                    );

                    if ($total_updates > 9 || count($core_updates_final)) {

                        $headers[] = 'X-Priority: 1 (Highest)';
                        $headers[] = 'X-MSMail-Priority: High';
                        $headers[] = 'Importance: High';

                    }

                    add_filter('wp_mail_from_name', 'wordpress_maintenance_class::wp_mail_from_name', 999999);

                    $admin_users = get_users('role=Administrator');
    				$admin_emails = array();

    				foreach ($admin_users as $admin_user) {

    					if (
    				    	$admin_user->user_email &&
    						!in_array(strtolower($admin_user->user_email), $admin_emails, true)
    					) {

    						$admin_emails[] = strtolower($admin_user->user_email);

    					}

    				}

    				if (
    					get_bloginfo('admin_email') &&
    					!in_array(strtolower(get_bloginfo('admin_email')), $admin_emails, true)
    				) {

    				    $admin_emails[] = strtolower(get_bloginfo('admin_email'));

    				}

    			    if ($admin_emails) {

    					if (is_multisite()) {

    						$blogname = get_network()->site_name;

    					} else {

    			    		$blogname = wp_specialchars_decode(get_option('blogname'), ENT_QUOTES);

    					}

    					foreach ($admin_emails as $admin_email) {

                            wp_mail(
                                $admin_email,
                                wp_specialchars_decode(sprintf(__('[%s] WordPress Service Required', 'wordpress-maintenance'), $blogname)),
                                $message,
                                $headers
                            );

    					}

    				}

                    remove_filter('wp_mail_from_name', 'wordpress_maintenance_class::wp_mail_from_name', 999999);

                    update_option('webd_uk_service_last_notified', time(), false);

                } elseif (!absint(get_option('webd_uk_service_last_notified'))) {

                    update_option('webd_uk_service_last_notified', strtotime('-6 months'), false);

                }

            }

        }

        public function wp_dashboard_setup() {

            wp_add_dashboard_widget('wordpress_maintenance_widget', 'WordPress Maintenance', array($this, 'wm_dashboard_widget'), null, null, 'normal', 'high');

        }

        public function wm_dashboard_widget() {

            $sys_loadavg = $this->load_average;

            if ($this->load_average) {

                if ($this->cores_available) {

                    foreach ($sys_loadavg as $key => $loadavg) {

                        $loadavg_color = $loadavg / $this->cores_available;

                        if ($loadavg_color > 1) {

                            $loadavg_color = 1;

                        }

                        $loadavg_color = ceil(abs($loadavg_color * 200));
                        $sys_loadavg[$key] = '<span style="color: #' . str_pad(dechex($loadavg_color), 2, '0', STR_PAD_LEFT) . str_pad(dechex(200 - $loadavg_color), 2, '0', STR_PAD_LEFT) . '00;">' . $loadavg . '</span> (' . array('1 minute', '5 minutes', '15 minutes')[$key] . ')';

                    }

                }

?>
<h2>Load Average</h2>
<p>The load average represents the load on the server over three periods of time:</p>
<p><?php echo implode(', ', $sys_loadavg); ?><?php

                if ($this->cores_available && $this->cores_recommended) {

?>
<br />
<strong>Available CPUs / Cores:</strong> <?php echo esc_html($this->cores_available); ?></br />
<strong>Recommended CPUs / Cores:</strong> <span style="color: #<?php echo str_pad(dechex(($this->cores_recommended > $this->cores_available ? 200 : 0)), 2, '0', STR_PAD_LEFT) . str_pad(dechex(200 - ($this->cores_recommended > $this->cores_available ? 200 : 0)), 2, '0', STR_PAD_LEFT); ?>00;"><?php echo $this->cores_recommended; ?></span> (<?php echo ($this->cores_recommended > $this->cores_available ? 'your server is overloaded' : 'your server is looking good'); ?>)<?php

                    if ($this->cores_recommended > $this->cores_available) {

                        if ($this->cores_available > 12) {

?>
<br />
It looks like you are on a cloud server so you might need to contact your host to upgrade to a VPS server if this persists<?php

                        } else {

?>
<br />
It looks like you are on a dedicated server so you might need to contact your host to upgrade the server if this persists<?php

                        }

                    }

                }

?></p>
<?php

            }

            if (wp_using_ext_object_cache() && method_exists('Memcache','getStats')) {

                $memcache_obj = new Memcache;
                $memcache_obj->addServer('127.0.0.1', '11211');
                $memcache_stats = $memcache_obj->getStats();

                if ($memcache_stats) {

?>
<h2>Memcached</h2>
<p style="color: #<?php echo str_pad(dechex((int) ($memcache_stats['bytes'] / $memcache_stats['limit_maxbytes'] * 200)), 2, '0', STR_PAD_LEFT) . str_pad(dechex((int) (200 - ($memcache_stats['bytes'] / $memcache_stats['limit_maxbytes'] * 200))), 2, '0', STR_PAD_LEFT); ?>00;">Cache is <?php echo absint($memcache_stats['bytes'] / $memcache_stats['limit_maxbytes'] * 100); ?>% used (<?php echo esc_html(self::readable_size($memcache_stats['bytes'])); ?> of <?php echo esc_html(self::readable_size($memcache_stats['limit_maxbytes'])); ?>)<?php

                    if ($memcache_stats['bytes'] / $memcache_stats['limit_maxbytes'] > 0.75) {

?><br />
Cache is approaching capactiy and you should consider increasing the Memcached memory<?php

                    }

?></p>
<?php

?>
<p><span class="wordpress-maintenance-flush-persistent-object-cache button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>" title="Flush Cache">Flush Cache</a></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-flush-persistent-object-cache').click(function() {
        if (confirm('<?php _e('Are you sure you want to flush the persistent object cache?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-flush-persistent-object-cache').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_flush_persistent_object_cache',
            	_ajax_nonce: '<?php echo wp_create_nonce('flush-persistent-object-cache'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo admin_url(); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo admin_url(); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo admin_url(); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php

					}

                }

            }

            $cloudflare_api_token = sanitize_text_field(get_option('wm_cloudflare_api_token'));

            if ($cloudflare_api_token && isset($_GET['check_cloudflare_zones'])) {

                $cloudflare_zones = self::get_cloudflare_zones($cloudflare_api_token);
				$cloudflare_zones_security_level = 'medium';

                if (is_array($cloudflare_zones) && $cloudflare_zones) {

?>
<h2>Cloudflare</h2>
<p>Cloudflare Under Attack Mode is: <?php echo (self::check_under_attack_file() ? '<span style="color:red">ON</span>' : '<span style="color:green">OFF</span>'); ?></p>
<p><strong>Security Levels</strong></p>
<?php

                    foreach ($cloudflare_zones as $cloudflare_zone) {

                        $cloudflare_zone_security_level = self::get_cloudflare_zone_security_level($cloudflare_api_token, $cloudflare_zone->id);

                        if ($cloudflare_zone_security_level) {

?>
<p>Zone <?php echo esc_html($cloudflare_zone->name); ?>: <?php echo esc_html($cloudflare_zone_security_level); ?></p>
<?php

							if ('under_attack' === $cloudflare_zone_security_level) {

								$cloudflare_zones_security_level = 'under_attack';

							}

                        }

                    }

?>
<p><span class="wordpress-maintenance-set-cloudflare-security-level button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>" title="<?php echo ('under_attack' === $cloudflare_zones_security_level ? 'Disable' : 'Enable'); ?> Under Attack Mode"><?php echo ('under_attack' === $cloudflare_zones_security_level ? 'Disable' : 'Enable'); ?> Under Attack Mode</span></p>
<?php if ($this->request_permission()) { ?><script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-set-cloudflare-security-level').click(function() {
        if (confirm('<?php _e('Are you sure you want to ' . ('under_attack' === $cloudflare_zones_security_level ? 'disable' : 'enable') . ' under attack mode?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-set-cloudflare-security-level').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_set_cloudflare_security_level',
            	_ajax_nonce: '<?php echo wp_create_nonce('set-cloudflare-security-level'); ?>',
            	value: '<?php echo ('under_attack' === $cloudflare_zones_security_level ? 'medium' : 'under_attack'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        window.location.href = '<?php echo add_query_arg('check_cloudflare_zones', 'true', admin_url()); ?>';
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('check_cloudflare_zones', 'true', admin_url()); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo add_query_arg('check_cloudflare_zones', 'true', admin_url()); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script><?php

					}

                }

            } elseif ($cloudflare_api_token) {

?>
<h2>Cloudflare</h2>
<p>Cloudflare Under Attack Mode is: <?php echo (self::check_under_attack_file() ? '<span style="color:red">ON</span>' : '<span style="color:green">OFF</span>'); ?></p>
<p><a href="<?php echo esc_url(add_query_arg('check_cloudflare_zones', 'true', admin_url())); ?>" class="button button-small<?php if (!$this->request_permission()) { echo ' disabled'; } ?>" title="Check Cloudflare Security Levels">Check Cloudflare Security Levels</a></p>
<?php

            }

?>
<h2>Scan WordPress</h2>
<p>Check to see if your WordPress website needs a service ...</p>
<p><a class="button button-small" href="https://webd.uk/scan/?url=<?php echo urlencode(home_url()); ?>" title="Scan WordPress" target="_blank">Scan now ...</a></p>
<?php

            if (!get_option('webd_uk_service_notifications') && $this->request_permission()) {

?>
<p><span class="wordpress-maintenance-enable-service-notifications button button-small"><?php _e('Enable Service Email Notifications', 'wordpress-maintenance'); ?></span></p>
<script type="text/javascript">
(function($) {
    $('.wordpress-maintenance-enable-service-notifications').click(function() {
        if (confirm('<?php _e('Are you sure you want to enable service notifications?', 'wordpress-maintenance'); ?>')) {
            $('.wordpress-maintenance-enable-service-notifications').unbind('click')
            var data = {
            	action: 'wordpress_maintenance_enable_service_notifications',
            	_ajax_nonce: '<?php echo wp_create_nonce('enable-service-notifications'); ?>'
            };
    	    $.ajax({
        	    url: ajaxurl,
        	    data: data,
                type: 'POST',
                success: function(response) {
                    if ('success' in response && response.success) {
                        $('.wordpress-maintenance-enable-service-notifications').remove();
                    } else {
                        alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                        window.location.href = '<?php echo admin_url(); ?>';
                    }
                },
                error: function() {
                    alert('<?php _e('Something went wrong!', 'wordpress-maintenance'); ?>');
                    window.location.href = '<?php echo admin_url(); ?>';
                }
    	    });
        }
    });
})(jQuery);
</script>
<?php

            }

        }

        public static function readable_size($size){

            $base = log($size) / log(1024);
            $suffix = array(' bytes', 'Kb', 'Mb', 'Gb', 'Tb');
            $f_base = floor($base);

            return round(pow(1024, $base - floor($base)), 1) . $suffix[$f_base];

        }

        public static function add_option($option, $value) {

            if ('active_options' !== $option) {

                $active_options = wp_cache_get('active_options');

                if (!$active_options || !is_array($active_options)) {

                    $active_options = array();

                }

                if (!in_array($option, $active_options, true)) {

                    $active_options[] = $option;

                }

                wp_cache_set('active_options', $active_options);

            }

        }

        public static function pre_option($pre, $option, $default) {

            if ('active_options' !== $option) {

                $active_options = wp_cache_get('active_options');

                if (!$active_options || !is_array($active_options)) {

                    $active_options = array();

                }

                if (!in_array($option, $active_options, true)) {

                    $active_options[] = $option;

                }

                wp_cache_set('active_options', $active_options);

            }

            return $pre;

        }

        public function shutdown() {

            global $wpsc;

            if (
                isset($wpsc) &&
                method_exists($wpsc, 'purge') &&
                $this->content_changed
            ) {

error_log('Purged Stack Cache');

                $wpsc::purge('all');

            }

            if (get_option('track_active_options')) {

                $active_options = wp_cache_get('active_options');

                if ($active_options && is_array($active_options)) {

                    $previous_active_options = get_option('active_options');

                    if ($previous_active_options && is_array($previous_active_options)) {

                        $active_options = array_unique(array_merge($active_options, $previous_active_options));

                    }

                    if (!$previous_active_options || (is_array($previous_active_options) && count($previous_active_options) !== count($active_options))) {

                        update_option('active_options', $active_options);

                    }

                }

            }

        }

        public function wordpress_maintenance_delete_all_transients() {

            if (
                isset($_POST['transient_type']) &&
                in_array($_POST['transient_type'], array('all', 'autoloaded'), true) &&
                $this->request_permission() && current_user_can('manage_options')
            ) {

                check_ajax_referer('delete-all-transients');

                $autoloaded = ('autoloaded' === $_POST['transient_type'] ? ' AND autoload = \'yes\'' : '');

                global $wpdb;
                $wpdb->query("DELETE FROM $wpdb->options WHERE option_name LIKE '_transient_%'$autoloaded;");
                $wpdb->query("DELETE FROM $wpdb->options WHERE option_name LIKE '_site_transient_%'$autoloaded;");

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_delete_oembed_cache_postmeta() {

            if (
                $this->request_permission() && current_user_can('manage_options')
            ) {

                check_ajax_referer('delete-oembed-cache-postmeta');

                global $wpdb;
                $wpdb->query("DELETE FROM $wpdb->postmeta WHERE meta_key LIKE '%_oembed_%';");

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_download_orphaned_meta_data() {

            global $wpdb;

            $orphaned_meta_checks = array(
                $wpdb->commentmeta => array(
                    'table' => $wpdb->comments,
                    'id' => 'comment_ID',
                    'meta_id' => 'meta_id',
                    'meta_object_id' => 'comment_id'
                ),
                $wpdb->postmeta => array(
                    'table' => $wpdb->posts,
                    'id' => 'ID',
                    'meta_id' => 'meta_id',
                    'meta_object_id' => 'post_id'
                ),
                $wpdb->termmeta => array(
                    'table' => $wpdb->terms,
                    'id' => 'term_id',
                    'meta_id' => 'meta_id',
                    'meta_object_id' => 'term_id'
                ),
                $wpdb->usermeta => array(
                    'table' => $wpdb->users,
                    'id' => 'ID',
                    'meta_id' => 'umeta_id',
                    'meta_object_id' => 'user_id'
                ),
            );

            if (
                isset($_GET['table']) &&
                isset($orphaned_meta_checks[sanitize_key($_GET['table'])]) &&
                $this->request_permission() && current_user_can('manage_options')
            ) {

                $meta_table = sanitize_key($_GET['table']);
                $orphaned_meta_check = $orphaned_meta_checks[$meta_table];

                check_ajax_referer('download-orphaned-meta-data-' . $meta_table);

                $meta_data = $wpdb->get_results("
SELECT {$orphaned_meta_check['meta_id']}, {$orphaned_meta_check['meta_object_id']}, meta_key, meta_value
FROM $meta_table
LEFT JOIN {$orphaned_meta_check['table']} ON $meta_table.{$orphaned_meta_check['meta_object_id']} = {$orphaned_meta_check['table']}.{$orphaned_meta_check['id']}
WHERE {$orphaned_meta_check['table']}.{$orphaned_meta_check['id']} IS NULL;
");

                if ($meta_data) {

                    header('Content-Type: application/json');
                    header('Content-Disposition: attachment;filename=' . sanitize_title_with_dashes(strtolower($meta_table)) . '-orphaned-meta-data.json');

                    echo json_encode($meta_data);

                } else {

                    check_ajax_referer('Forcing a 403 error');

                }

            } else {

                check_ajax_referer('Forcing a 403 error');

            }

            exit;

        }

        public function wordpress_maintenance_delete_orphaned_meta_data() {

            global $wpdb;

            $orphaned_meta_checks = array(
                $wpdb->commentmeta => array(
                    'table' => $wpdb->comments,
                    'id' => 'comment_ID',
                    'meta_id' => 'meta_id',
                    'meta_object_id' => 'comment_id'
                ),
                $wpdb->postmeta => array(
                    'table' => $wpdb->posts,
                    'id' => 'ID',
                    'meta_id' => 'meta_id',
                    'meta_object_id' => 'post_id'
                ),
                $wpdb->termmeta => array(
                    'table' => $wpdb->terms,
                    'id' => 'term_id',
                    'meta_id' => 'meta_id',
                    'meta_object_id' => 'term_id'
                ),
                $wpdb->usermeta => array(
                    'table' => $wpdb->users,
                    'id' => 'ID',
                    'meta_id' => 'umeta_id',
                    'meta_object_id' => 'user_id'
                ),
            );

            if (
                isset($_POST['table']) &&
                isset($orphaned_meta_checks[sanitize_key($_POST['table'])]) &&
                $this->request_permission() && current_user_can('manage_options')
            ) {

                check_ajax_referer('delete-orphaned-meta-data');

                $wpdb->query("
DELETE " . sanitize_key($_POST['table']) . " FROM " . sanitize_key($_POST['table']) . "
LEFT JOIN {$orphaned_meta_checks[sanitize_key($_POST['table'])]['table']} ON " . sanitize_key($_POST['table']) . ".{$orphaned_meta_checks[sanitize_key($_POST['table'])]['meta_object_id']} = {$orphaned_meta_checks[sanitize_key($_POST['table'])]['table']}.{$orphaned_meta_checks[sanitize_key($_POST['table'])]['id']}
WHERE {$orphaned_meta_checks[sanitize_key($_POST['table'])]['table']}.{$orphaned_meta_checks[sanitize_key($_POST['table'])]['id']} IS NULL;
");

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function admin_notices() {

            $sys_loadavg = $this->load_average;

            if (
                $this->load_average &&
                $this->cores_available &&
                ($this->load_average[0] / $this->cores_available > 1) &&
                ($this->load_average[1] / $this->cores_available > 1)
            ) {

                if ($this->cores_available) {

                    foreach ($sys_loadavg as $key => $loadavg) {

                        $loadavg_color = $loadavg / $this->cores_available;

                        if ($loadavg_color > 1) {

                            $loadavg_color = 1;

                        }

                        $loadavg_color = ceil(abs($loadavg_color * 200));
                        $sys_loadavg[$key] = '<span style="color: #' . str_pad(dechex($loadavg_color), 2, '0', STR_PAD_LEFT) . str_pad(dechex(200 - $loadavg_color), 2, '0', STR_PAD_LEFT) . '00;">' . $loadavg . '</span> (' . array('1 minute', '5 minutes', '15 minutes')[$key] . ')';

                    }

                }

?>
<div class="notice notice-error">
<p>High load average: <?php echo implode(', ', $sys_loadavg); ?><?php

                if ($this->cores_available && $this->cores_recommended) {

?>
<br />
<strong>Available CPUs / Cores:</strong> <?php echo esc_html($this->cores_available); ?></br />
<strong>Recommended CPUs / Cores:</strong> <span style="color: #<?php echo str_pad(dechex(($this->cores_recommended > $this->cores_available ? 200 : 0)), 2, '0', STR_PAD_LEFT) . str_pad(dechex(200 - ($this->cores_recommended > $this->cores_available ? 200 : 0)), 2, '0', STR_PAD_LEFT); ?>00;"><?php echo $this->cores_recommended; ?></span> (<?php echo ($this->cores_recommended > $this->cores_available ? 'your server is overloaded' : 'your server is looking good'); ?>)<?php

                    if ($this->cores_recommended > $this->cores_available) {

                        if ($this->cores_available > 12) {

?>
<br />
It looks like you are on a cloud server so you might need to contact your host to upgrade to a VPS server if this persists.<?php

                        } else {

?>
<br />
It looks like you are on a dedicated server so you might need to contact your host to upgrade the server if this persists.<?php

                        }

                    }

                }

?></p>
</div>
<?php

            }

        }

        public static function upgrader_post_install($response, $hook_extra, $result) {

            if (wp_using_ext_object_cache() && method_exists('Memcache','getStats')) {

                $memcache_obj = new Memcache;
                $memcache_obj->addServer('127.0.0.1', '11211');
                $memcache_obj->flush();

            }

        }

        public static function cron_schedules($schedules) {

            $schedules['fifteen_minutes'] = array(
                'interval' => 900,
                'display' => esc_html__('Every Fifteen Minutes', 'wordpress-maintenance')
            );

            return $schedules;

        }

        public function wordpress_maintenance_set_cloudflare_api_token() {

            check_ajax_referer('set-cloudflare-api-token');

            if (
                $this->request_permission() &&
                current_user_can('manage_options') &&
                isset($_POST['cloudflare_api_token']) &&
                sanitize_text_field($_POST['cloudflare_api_token'])
            ) {

                update_option('wm_cloudflare_api_token', sanitize_text_field($_POST['cloudflare_api_token']));

                wp_send_json_success();

            } elseif(
                $this->request_permission() &&
                current_user_can('manage_options') &&
                isset($_POST['cloudflare_api_token'])
            ) {

                delete_option('wm_cloudflare_api_token');

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_quarter_hourly_schedule() {

            $sys_loadavg = $this->load_average;

            if (
                $this->load_average &&
                $this->cores_available &&
                ($this->load_average[0] / $this->cores_available > 1) &&
                ($this->load_average[1] / $this->cores_available > 1) &&
                !sanitize_text_field(get_option('wm_cloudflare_api_token'))
            ) {

                error_log('Server overloaded. Cloudflare auto-protect recommended. Available CPUs / Cores: ' . $this->cores_available . ' Recommended CPUs / Cores: ' . $this->cores_recommended . ' Load average: ' . $this->load_average[0] . ' ' . $this->load_average[1] . ' ' . $this->load_average[2]);

            }

        }

        public static function wordfence_security_event($event) {

            if (
                'increasedAttackRate' === $event &&
                $cloudflare_api_token = sanitize_text_field(get_option('wm_cloudflare_api_token'))
            ) {

                if (
                    !self::check_under_attack_file() &&
                    self::acquire_under_attack_lock()
                ) {

                    $message = __('Wordfence has detected an increased attack rate so Cloudflare "Under Attack" mode has been enabled.', 'wordpress-maintenance');
                    self::enable_cloudflare_under_attack_mode($message);
                    self::release_under_attack_lock(true);

                }

            }

        }

        private static function enable_cloudflare_under_attack_mode($message, $overloaded = false) {

            if (
                $cloudflare_api_token = sanitize_text_field(get_option('wm_cloudflare_api_token'))
            ) {

                $cloudflare_zones = self::get_cloudflare_zones($cloudflare_api_token);

                if (is_array($cloudflare_zones) && $cloudflare_zones) {

                    $result = self::set_cloudflare_under_attack($cloudflare_api_token, $cloudflare_zones);

                    if (!self::check_under_attack_file()) {

                        if ($result) {

                            $admin_emails = self::get_wordpress_admin_emails('wm_enable_cloudflare_notifications');

        		    		if ($admin_emails) {

            					if (is_multisite()) {

            						$blogname = get_network()->site_name;

            					} else {

            	    				$blogname = wp_specialchars_decode(get_option('blogname'), ENT_QUOTES);

            					}

            					foreach ($admin_emails as $admin_email) {

            						wp_mail(
            							$admin_email,
            							wp_specialchars_decode(sprintf(__('[%s] Cloudflare "Under Attack" mode enabled', 'wordpress-maintenance'), $blogname)),
            							$message
            						);

            					}

        		    		}

                            if (!self::create_under_attack_file($overloaded)) {

                                error_log('Could not create .under_attack');

                            }

    					}

                    }

                }

            }

        }

        private static function disable_cloudflare_under_attack_mode() {

            if (
                $cloudflare_api_token = sanitize_text_field(get_option('wm_cloudflare_api_token'))
            ) {

                $cloudflare_zones = self::get_cloudflare_zones($cloudflare_api_token);

                if (is_array($cloudflare_zones) && $cloudflare_zones) {

                    $result = self::set_cloudflare_under_attack($cloudflare_api_token, $cloudflare_zones, true);

                    $under_attack_status = self::check_under_attack_file();

                    if ($under_attack_status) {

                        if ($result) {

                            $admin_emails = self::get_wordpress_admin_emails('wm_enable_cloudflare_notifications');

        		    		if ($admin_emails) {

            					if (is_multisite()) {

            						$blogname = get_network()->site_name;

            					} else {

            	    				$blogname = wp_specialchars_decode(get_option('blogname'), ENT_QUOTES);

            					}

                                $time_difference = time() - $under_attack_status['timestamp'];

                                if ($time_difference > 59) {

                                    $time_difference = absint($time_difference / 60) . ' minute(s)';

                                } else {

                                    $time_difference = $time_difference . ' second(s)';

                                }

                                if (
                                    isset($under_attack_status['overloaded']) &&
                                    $under_attack_status['overloaded']
                                ) {

                                    $message = sprintf(__(
                                        '"Under Attack" mode was enabled for %s.
    
The website\'s server is no longer overloaded so Cloudflare "Under Attack" mode has been disabled.',
                                        'wordpress-maintenance'
                                    ), $time_difference);

                                } else {

                                    $message = sprintf(__(
                                        '"Under Attack" mode was enabled for %s.
    
The website\'s server is not overloaded so Cloudflare "Under Attack" mode has been disabled.',
                                        'wordpress-maintenance'
                                    ), $time_difference);

                                }

            					foreach ($admin_emails as $admin_email) {

            						wp_mail(
            							$admin_email,
            							wp_specialchars_decode(sprintf(__('[%s] Cloudflare "Under Attack" mode disabled', 'wordpress-maintenance'), $blogname)),
                                        $message
            						);

            					}

        					}

                            if (!self::delete_under_attack_file()) {

                                error_log('Could not delete .under_attack');

                            }

                        }

                    }

                }

            }

        }

        private static function get_wordpress_admin_emails($notification_type = '') {

    		$admin_users = get_users('role=Administrator');
    		$admin_emails = array();

    		foreach ($admin_users as $admin_user) {

    			if (
    				$admin_user->user_email &&
    				!in_array(strtolower($admin_user->user_email), $admin_emails, true) &&
    				(
        				!$notification_type ||
        				(
        				    $notification_type &&
        				    get_user_meta($admin_user->ID, $notification_type, true)
        				)
        			)
    			) {

    				$admin_emails[] = strtolower($admin_user->user_email);

				}

    		}

    		if (
    			get_bloginfo('admin_email') &&
                !$admin_emails
    		) {

    	   		$admin_emails[] = strtolower(get_bloginfo('admin_email'));

    		}

            return $admin_emails;

        }

        private static function set_cloudflare_under_attack($cloudflare_api_token, $cloudflare_zones, $disable = false) {

            if ($disable) {

                $security_level_to_set = 'medium';

            } else {

                $security_level_to_set = 'under_attack';

            }

            $failed_cloudflare_zones = array();

            foreach ($cloudflare_zones as $cloudflare_zone) {

                $cloudflare_zone_security_level = self::get_cloudflare_zone_security_level($cloudflare_api_token, $cloudflare_zone->id);

                if ($security_level_to_set !== $cloudflare_zone_security_level) {

					if (!self::set_cloudflare_zone_security_level($cloudflare_api_token, $cloudflare_zone->id, $security_level_to_set)) {

                        $failed_cloudflare_zones[] = $cloudflare_zone->name;

					}

                } else {

                    $failed_cloudflare_zones[] = $cloudflare_zone->name;

				}

            }

error_log('Cloudflare under attack mode ' . ($disable ? 'disabled' : 'set') . ($failed_cloudflare_zones ? ' (but failed for ' . implode(', ', $failed_cloudflare_zones) . ')' : '') . '.');

            return !((bool) $failed_cloudflare_zones);

        }

        private static function get_cloudflare_zones($cloudflare_api_token) {

            if (sanitize_text_field($cloudflare_api_token)) {

                $cloudflare_response = wp_remote_request(
                    'https://api.cloudflare.com/client/v4/zones',
                    array(
                        'method' => 'GET',
                        'headers' => array(
                            'Authorization' => 'Bearer ' . sanitize_text_field($cloudflare_api_token),
                            'Content-Type' => 'application/json'
                        )
                    )
                );

                if (
                    !is_wp_error($cloudflare_response) &&
                    isset($cloudflare_response['body'])
                ) {

                    $cloudflare_result = json_decode($cloudflare_response['body']);

                    if (0 === json_last_error()) {

                        if (
                            isset($cloudflare_result->result) &&
                            is_array($cloudflare_result->result) &&
                            $cloudflare_result->result
                        ) {

                            return $cloudflare_result->result;

                        } else {

error_log('Cloudflare zones not in response: ' . print_r($cloudflare_result->errors, true));

                        }

                    } else {

error_log('No json in Cloudflare zone request: ' . print_r($cloudflare_response, true));

                    }

                } else {

error_log('No body in Cloudflare zone response: ' . print_r($cloudflare_response, true));

                }

            }

            return false;

        }

        private static function get_cloudflare_zone_security_level($cloudflare_api_token, $cloudflare_zone_id) {

            if (sanitize_text_field($cloudflare_api_token) && sanitize_key($cloudflare_zone_id)) {

                $cloudflare_response = wp_remote_request(
                    'https://api.cloudflare.com/client/v4/zones/' . sanitize_key($cloudflare_zone_id) . '/settings/security_level',
                    array(
                        'method' => 'GET',
                        'headers' => array(
                            'Authorization' => 'Bearer ' . $cloudflare_api_token,
                            'Content-Type' => 'application/json'
                        )
                    )
                );

                if (
                    !is_wp_error($cloudflare_response) &&
                    isset($cloudflare_response['body'])
                ) {

                    $cloudflare_result = json_decode($cloudflare_response['body']);

                    if (0 === json_last_error()) {

                        if (isset($cloudflare_result->result->value) && $cloudflare_result->result->value) {

                            return $cloudflare_result->result->value;

                        } else {

error_log('Cloudflare zones security level not in response: ' . print_r($cloudflare_result->errors, true));

                        }

                    } else {

error_log('No json in Cloudflare security level request: ' . print_r($cloudflare_response, true));

                    }

                } else {

error_log('No body in Cloudflare security level request: ' . print_r($cloudflare_response, true));

                }

            } else {

error_log('Bad Cloudflare API token (' . $cloudflare_api_token . ') or zone ID (' . $cloudflare_zone_id . ')');

            }

            return false;

        }

        private static function get_cloudflare_zone_access_rules($cloudflare_api_token, $cloudflare_zone_id) {

            if (sanitize_text_field($cloudflare_api_token) && sanitize_key($cloudflare_zone_id)) {

                $cloudflare_response = wp_remote_request(
                    'https://api.cloudflare.com/client/v4/zones/' . sanitize_key($cloudflare_zone_id) . '/firewall/access_rules/rules',
                    array(
                        'method' => 'GET',
                        'headers' => array(
                            'Authorization' => 'Bearer ' . $cloudflare_api_token,
                            'Content-Type' => 'application/json'
                        )
                    )
                );

                if (
                    !is_wp_error($cloudflare_response) &&
                    isset($cloudflare_response['body'])
                ) {

                    $cloudflare_result = json_decode($cloudflare_response['body']);

                    if (0 === json_last_error()) {

                        if (isset($cloudflare_result->result) && $cloudflare_result->result) {

                            return $cloudflare_result->result;

                        } else {

error_log('Cannot get Cloudflare Zone Access Rules: Cloudflare zones access rules not in response.');

                        }

                    } else {

error_log('Cannot get Cloudflare Zone Access Rules: No json in Cloudflare access rules request.');

                    }

                } else {

error_log('Cannot get Cloudflare Zone Access Rules: No body in Cloudflare access rules request.');

                }

            } else {

error_log('Cannot get Cloudflare Zone Access Rules: Bad Cloudflare API token zone ID.');

            }

            return false;

        }

        private static function set_cloudflare_zone_security_level($cloudflare_api_token, $cloudflare_zone_id, $value) {

            if (sanitize_text_field($cloudflare_api_token) && sanitize_key($cloudflare_zone_id)) {

                self::set_cloudflare_zone_access_rule($cloudflare_api_token, $cloudflare_zone_id);

				$cloudflare_response = wp_remote_request(
					'https://api.cloudflare.com/client/v4/zones/' . sanitize_key($cloudflare_zone_id) . '/settings/security_level',
					array(
						'method' => 'PATCH',
						'headers' => array(
							'Authorization' => 'Bearer ' . $cloudflare_api_token,
							'Content-Type' => 'application/json'
						),
						'body' => json_encode(array('value' => $value))
					)
				);

                if (
                    !is_wp_error($cloudflare_response) &&
                    isset($cloudflare_response['body'])
                ) {

                    $cloudflare_result = json_decode($cloudflare_response['body']);

                    if (0 === json_last_error()) {

                        if (isset($cloudflare_result->result->value) && $cloudflare_result->result->value) {

                            return $cloudflare_result->result->value;

                        } else {

error_log('Cloudflare zones security level not in response: ' . print_r($cloudflare_result->errors, true));

                        }

                    } else {

error_log('No json in Cloudflare security level request: ' . print_r($cloudflare_response, true));

                    }

                } else {

error_log('No body in Cloudflare security level request: ' . print_r($cloudflare_response, true));

                }

            } else {

error_log('Bad Cloudflare API token (' . $cloudflare_api_token . ') or zone ID (' . $cloudflare_zone_id . ')');

            }

            return false;

        }

        private static function set_cloudflare_zone_access_rule($cloudflare_api_token, $cloudflare_zone_id) {

            if (sanitize_text_field($cloudflare_api_token) && sanitize_key($cloudflare_zone_id)) {

                $cloudflare_response = array();
                $public_ip_request = wp_remote_get('https://ipinfo.io/ip');
                $public_ip = '';
                $rule_found = false;
                $ip_found = false;

                if (
                    isset($public_ip_request['body']) &&
                    $public_ip_request['body'] &&
                    (
                        filter_var($public_ip_request['body'], FILTER_VALIDATE_IP, FILTER_FLAG_IPV4) ||
                        filter_var($public_ip_request['body'], FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)
                    )
                ) {

                    $public_ip = $public_ip_request['body'];

                } else {

error_log('Cannot get public IP address: ' . print_r($public_ip_request, true));

                    return false;

                }

                $cloudflare_zone_access_rules = self::get_cloudflare_zone_access_rules($cloudflare_api_token, $cloudflare_zone_id);

                if ($cloudflare_zone_access_rules && is_array($cloudflare_zone_access_rules)) {

                    foreach ($cloudflare_zone_access_rules as $key => $cloudflare_zone_access_rule) {

                        if (
                            isset($cloudflare_zone_access_rule->configuration->value) &&
                            $public_ip === $cloudflare_zone_access_rule->configuration->value
                        ) {

                            $ip_found = true;
                            $rule_found = $cloudflare_zone_access_rule->id;

                            break;

                        }

                        if (
                            isset($cloudflare_zone_access_rule->notes) &&
                            'Whitelisted server IP via WordPress Maintenance plugin.' === $cloudflare_zone_access_rule->notes
                        ) {

                            $rule_found = $cloudflare_zone_access_rule->id;

                        }

                    }

                    if ($ip_found && $rule_found) {

                        return true;

                    } elseif ($rule_found) {

        				$cloudflare_response = wp_remote_request(
        					'https://api.cloudflare.com/client/v4/zones/' . sanitize_key($cloudflare_zone_id) . '/firewall/access_rules/rules',
        					array(
        						'method' => 'DELETE',
        						'headers' => array(
        							'Authorization' => 'Bearer ' . $cloudflare_api_token,
        							'Content-Type' => 'application/json'
        						),
        						'body' => json_encode(array(
        						    'id' => $rule_found
	        				    ))
	        				)
	        			);

                        if (
                            !is_wp_error($cloudflare_response) &&
                            isset($cloudflare_response['body'])
                        ) {

                            $cloudflare_result = json_decode($cloudflare_response['body']);

                            if (0 === json_last_error()) {

                                if ((isset($cloudflare_result->result) && $cloudflare_result->result)) {

error_log('Old Cloudflare zone access rule removed.');

                                } else {

error_log('Cloudflare zone access rule delete request fail: ' . print_r($cloudflare_result->errors, true));

                                }

                            } else {

error_log('No json in Cloudflare access rules delete request: ' . print_r($cloudflare_response, true));

                            }

                        } else {

error_log('No body in Cloudflare access rules delete request: ' . print_r($cloudflare_response, true));

                        }

                    }

                }

                if (filter_var($public_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {

    				$cloudflare_response = wp_remote_request(
    					'https://api.cloudflare.com/client/v4/zones/' . sanitize_key($cloudflare_zone_id) . '/firewall/access_rules/rules',
    					array(
    						'method' => 'POST',
    						'headers' => array(
    							'Authorization' => 'Bearer ' . $cloudflare_api_token,
    							'Content-Type' => 'application/json'
    						),
    						'body' => json_encode(array(
    						    'configuration' => array(
        					        'target' => 'ip',
                                    'value' => filter_var($public_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)
        				        ),
        					    'mode' => 'whitelist',
        					    'notes' => 'Whitelisted server IP via WordPress Maintenance plugin.'
        				    ))
        				)
        			);

                } elseif (filter_var($public_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {

    				$cloudflare_response = wp_remote_request(
    					'https://api.cloudflare.com/client/v4/zones/' . sanitize_key($cloudflare_zone_id) . '/firewall/access_rules/rules',
    					array(
	       					'method' => 'POST',
    						'headers' => array(
    							'Authorization' => 'Bearer ' . $cloudflare_api_token,
    							'Content-Type' => 'application/json'
    						),
    						'body' => json_encode(array(
    						    'configuration' => array(
                                    'target' => 'ip6',
                                    'value' => filter_var($public_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)
    					        ),
    						    'mode' => 'whitelist',
    						    'notes' => 'Whitelisted server IP via WordPress Maintenance plugin.'
    					    ))
    					)
    				);

                }

                if (
                    !is_wp_error($cloudflare_response) &&
                    isset($cloudflare_response['body'])
                ) {

                    $cloudflare_result = json_decode($cloudflare_response['body']);

                    if (0 === json_last_error()) {

                        if (isset($cloudflare_result->result) && $cloudflare_result->result) {

error_log('Cloudflare zone access rule set.');

                            return true;

                        } else {

error_log('Cloudflare zone access rules not in response: ' . print_r($cloudflare_result->errors, true));

                        }

                    } else {

error_log('No json in Cloudflare access rules request: ' . print_r($cloudflare_response, true));

                    }

                } else {

error_log('Could not get public IP address or no body in Cloudflare access rules request: ' . print_r($cloudflare_response, true));

                }

            } else {

error_log('Bad Cloudflare API token (' . $cloudflare_api_token . ') or zone ID (' . $cloudflare_zone_id . ')');

            }

            return false;

        }

        public function wordpress_maintenance_set_cloudflare_security_level() {

            check_ajax_referer('set-cloudflare-security-level');

            if ($this->request_permission() && current_user_can('manage_options') && isset($_POST['value']) && in_array($_POST['value'], array('medium', 'under_attack'), true)) {

                $failed_cloudflare_zones = array();
				$cloudflare_zones_security_level = sanitize_key($_POST['value']);
                $cloudflare_api_token = sanitize_text_field(get_option('wm_cloudflare_api_token'));

                if ($cloudflare_api_token) {

                    $cloudflare_zones = self::get_cloudflare_zones($cloudflare_api_token);

                    if (is_array($cloudflare_zones) && $cloudflare_zones) {

                        foreach ($cloudflare_zones as $cloudflare_zone) {

                            $cloudflare_zone_security_level = self::get_cloudflare_zone_security_level($cloudflare_api_token, $cloudflare_zone->id);

                            if ($cloudflare_zones_security_level !== $cloudflare_zone_security_level) {

								if (!self::set_cloudflare_zone_security_level($cloudflare_api_token, $cloudflare_zone->id, $cloudflare_zones_security_level)) {

                                    $failed_cloudflare_zones[] = $cloudflare_zone->name;

									wp_send_json_error('Failed to set Cloudflare security level to ' . $cloudflare_zones_security_level . ' for zone: ' . $cloudflare_zone->name);

								}

                            }

                        }

error_log('Cloudflare security level set to ' . $cloudflare_zones_security_level . ($failed_cloudflare_zones ? ' (but failed for ' . implode(', ', $failed_cloudflare_zones) . ')' : '') . '.');

                    } else {

                        $failed_cloudflare_zones[] = 'all';

error_log('No Cloudflare zones to set Cloudflare security level to ' . $cloudflare_zones_security_level . '.');

						wp_send_json_error('No Cloudflare zones to set Cloudflare security level to ' . $cloudflare_zones_security_level . '.');

                    }

                } else {

error_log('No Cloudflare API token to set Cloudflare security level to ' . $cloudflare_zones_security_level . '.');

					wp_send_json_error('No Cloudflare API token to set Cloudflare security level to ' . $cloudflare_zones_security_level . '.');

                }

				if (
				    'medium' === $cloudflare_zones_security_level &&
				    !$failed_cloudflare_zones
				) {

                    if (!self::delete_under_attack_file()) {

                        error_log('Could not delete .under_attack');

                    }

				} elseif ('under_attack' === $cloudflare_zones_security_level) {

                    if (!self::create_under_attack_file()) {

                        error_log('Could not create .under_attack');

                    }


				}

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function wordpress_maintenance_flush_persistent_object_cache() {

            check_ajax_referer('flush-persistent-object-cache');

            if ($this->request_permission() && current_user_can('manage_options')) {

                if (wp_using_ext_object_cache() && method_exists('Memcache','getStats')) {

                    $memcache_obj = new Memcache;
                    $memcache_obj->addServer('127.0.0.1', '11211');
                    $memcache_obj->flush();

                }

                wp_send_json_success();

            } else {

                wp_send_json_error();

            }

        	wp_die();

        }

        public function edit_user_profile($user) {

            if (
                current_user_can('manage_options') &&
                get_option('wm_cloudflare_api_token')
            ) {

                wp_nonce_field('enable_cloudflare_notifications', '_wm_nonce');

?>
<h3>WordPress Maintenance</h3>
<table class="form-table">
    <tbody><tr>
        <th><label for="my_checkbox_field">Enable Cloudflare Notifications</label></th>
        <td>
            <input type="checkbox" name="wm_enable_cloudflare_notifications" id="wm_enable_cloudflare_notifications" value="1" <?php checked(get_user_meta($user->ID, 'wm_enable_cloudflare_notifications', true), '1' ); ?> />
            <label for="wm_enable_cloudflare_notifications"> Receive notifications when Cloudflare "Under Attack" mode is enabled or disabled.</label>
        </td>
    </tr></tbody>
</table>
<?php

            }

        }

        public function edit_user_profile_update($user_id) {

            if (
                isset($_POST['_wm_nonce']) &&
                wp_verify_nonce($_POST['_wm_nonce'], 'enable_cloudflare_notifications') && 
                current_user_can('manage_options') &&
                sanitize_text_field(get_option('wm_cloudflare_api_token'))
            ) {

                if (!isset($_POST['wm_enable_cloudflare_notifications'])) {

                    delete_user_meta( $user_id, 'wm_enable_cloudflare_notifications');

                } else {

                    update_user_meta($user_id, 'wm_enable_cloudflare_notifications', '1');

                }

            }

        }

	}

    if (!class_exists('wmCommon')) {

        require_once(dirname(__FILE__) . '/includes/class-wm-common.php');

    }

    if (!class_exists('wmPremium')) {

        require_once(dirname(__FILE__) . '/includes/class-wm-premium.php');

    }

	$wordpress_maintenance_object = new wordpress_maintenance_class();

    function is_server_overloaded() {

        global $wordpress_maintenance_object;

        return $wordpress_maintenance_object->server_overloaded;

    }

}

?>
